define({ "api": [
  {
    "type": "post",
    "url": "http://127.0.0.1:51088/login",
    "title": "Login",
    "name": "LoginQuery",
    "group": "Authentication",
    "version": "1.0.0",
    "description": "<p>This method performs user authentication (verifies user name and password) and afterwards user authorization (checks user permissions to use REST API methods).<br> The user credentials should be provided in the HTTP body (in JSON format). The password must be encrypted like described below. The Switch Web Services decrypt and verify the password and if the password matches the user name, the Switch Web Services also check the user permissions.<br> If the login is successful, the Switch Web Services generate a unique bearer token (a string like &quot;ab43e62fbbd5ea541efbbb79c74e5d1c7411c6f6&quot;) for the user and send it back as a part of the login response. The response also includes the basic information about the permissions the user has. If the login fails, the Switch Web Services respond with an error status code and optionally with the error message like described below.<br> As soon as the client receives this token, it is able to call all other REST API methods it has access to. The token must be put in the HTTP Authorization header for each subsequent REST call, so the header looks like: <i>Authorization: Bearer ab43e62fbbd5ea541efbbb79c74e5d1c7411c6f6</i><br> This token is used by the Switch Web Services to understand which user performs the REST call, so the Web Services can also verify if that user has the permission to use this call.</p>",
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "username",
            "description": "<p>User name</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "password",
            "description": "<p>User password encrypted by the RSA algorithm with padding PKCS1, then converted to base64 and preceded by '!@$'<br> The following public key is used for RSA encryption:<br> -----BEGIN PUBLIC KEY-----<br> MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC/ZW93qx1U6iGDwFWjYRpgS6/l<br> vFV/VGEM7wV14Gr4ZZxpZto3rpot3As4Cn0a++LwVh1oSidmiRWnHcKBDCmov8IL<br> fJ6SckFV41WZWibx8eSTCfn6zlrQ6QgJC2jdpLbWewxhox58zVXk5iK67GOKqbE/<br> HzSFJVdqIyjNX0/IpQIDAQAB<br> -----END PUBLIC KEY----- <br> This is an example of an encrypted empty password (default password for user Administrator): <br> !@$QBk%2BmdP1YJcU2cf61053cbstHmB7ysxyFB2Oy63WZwtSWN%2FhJzfVuiYo%2Fn%2Fx2Xn4x7f18%2BOB8oc1jovP%2FkNqMjj2e5o3XZYGWE7Qp6%2BaREcnu4XbFuooJ3y0lKPfC%2B0oDGq1P368uFdB7PznwhtOEKLUCl1CrNz98zzt1avqXgY%3D</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"password\" : \"!@$QBk%2BmdP1YJ...\",\n  \"username\" : \"Administrator\"\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "success",
            "description": "<p>Authentication result.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "user",
            "description": "<p>User name.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "token",
            "description": "<p>Bearer token, which has to be used in all the subsequent requests like it is shown above.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "jobClientAccess",
            "description": "<p>Permission to check and submit jobs, allowed or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "messagesAccess",
            "description": "<p>Permission to view messages, allowed or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "jobFinderAccess",
            "description": "<p>Permission to create boards, allowed or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "viewBoardsAccess",
            "description": "<p>Permission to view boards, allowed or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "reportingAccess",
            "description": "<p>Permission to create dashboards, allowed or not. Always true if the Reporting Module is activated.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "webservicesAccess",
            "description": "<p>Web services module, activated or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "administratorAccess",
            "description": "<p>Permission to perform administrative tasks, allowed or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "rushJobsAccess",
            "description": "<p>Permission to rush jobs, allowed or not.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"success\": true,\n  \"user\": \"Administrator\",\n  \"token\": \"b767c71faba9ff7b54e530a0cbf996ddb315131d\",\n  \"jobClientAccess\": true,\n  \"messagesAccess\": true,\n  \"jobFinderAccess\": true,\n  \"viewBoardsAccess\": true,\n  \"reportingAccess\": true,\n  \"webservicesAccess\": true\n  \"administratorAccess\": true\n  \"rushJobsAccess\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 403": [
          {
            "group": "Error 403",
            "type": "Boolean",
            "optional": false,
            "field": "success",
            "description": "<p>Always false</p>"
          },
          {
            "group": "Error 403",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message</p>"
          }
        ],
        "Error 401": [
          {
            "group": "Error 401",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 500": [
          {
            "group": "Error 500",
            "type": "Boolean",
            "optional": false,
            "field": "success",
            "description": "<p>Always false</p>"
          },
          {
            "group": "Error 500",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  \"success\": false,\n  \"error\": 'Invalid username or password.'\n}",
          "type": "json"
        },
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"success\": false,\n  \"error\": 'Cannot login: authentication error.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/login"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:51088/login' -H 'Content-Type: application/json' --data '{\"username\": \"Administrator\", \"password\": \"!%40%24QBk%252BmdP1YJcU2cf61053cbstHmB7ysxyFB2Oy63WZwtSWN%252FhJzfVuiYo%252Fn%252Fx2Xn4x7f18%252BOB8oc1jovP%252FkNqMjj2e5o3XZYGWE7Qp6%252BaREcnu4XbFuooJ3y0lKPfC%252B0oDGq1P368uFdB7PznwhtOEKLUCl1CrNz98zzt1avqXgY%253D\"}'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/webmain.js",
    "groupTitle": "Authentication"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/logout",
    "title": "Logout",
    "version": "1.0.0",
    "name": "LogoutQuery",
    "group": "Authentication",
    "description": "<p>Logout of a currently logged in user</p>",
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 401": [
          {
            "group": "Error 401",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 500": [
          {
            "group": "Error 500",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false</p>"
          },
          {
            "group": "Error 500",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Errror:\n{\n  \"status\": false,\n  \"error\": 'Cannot logout user. Internal server error.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/logout"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/logout/' -H 'Authorization: Bearer 6706ee088a9008e5430c50ca4e7ca1f6ecb53e63'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/webmain.js",
    "groupTitle": "Authentication"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:51088/api/v1/flows/:id/annotations",
    "title": "Add annotations",
    "name": "AddAnnotations",
    "group": "Flows",
    "version": "1.0.0",
    "description": "<p>This method allows you to add new annotations for particular flow.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          },
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-type",
            "description": "<p>application/json</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Content-type\": \"application/json\"\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Object[]",
            "optional": false,
            "field": "annotations",
            "description": "<p>Annotation object or array of such objects.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "annotations.type",
            "description": "<p>Type of annotation (one of the following values: &quot;info&quot;, &quot;question&quot;, &quot;warning&quot;, &quot;comment&quot;)</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "annotations.title",
            "description": "<p>Title of the annotation</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "annotations.description",
            "description": "<p>Description of the annotation</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": false,
            "field": "annotations.x",
            "description": "<p>The annotation's x-coordinate on canvas</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": false,
            "field": "annotations.y",
            "description": "<p>The annotation's y-coordinate on canvas</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>URL parameter. ID of the flow.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN</b>. If not specified, enUS will be used.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "[\n  {\"type\" : \"question\", \"title\" : \"some title\", \"description\" : \"description\", \"x\" : 7, \"y\" : 7 },\n  {\"type\" : \"info\", \"title\" : \"some title\", \"description\" : \"description\", \"x\" : 7, \"y\" : 8 }\n]",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: Annotations were successfully added. <br> <b> false</b>: Something went wrong.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": false,
            "field": "result",
            "description": "<p>Result of the operation. The format is the same as for request body with some additional fields, which were added by Switch: &quot;user&quot;(person who created the annotation), &quot;id&quot; (unique ID of the annotation), &quot;modified&quot; and &quot;created&quot; (dates in ISO format)</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true,\n  result: [\n    {\"type\" : \"question\", \"title\" : \"some title\", \"description\" : \"description\", \"x\" : 7, \"y\" : 7, \"id\": \"e43242-rewrqew-fndk423ds-jfe3xgdo\", \"modified\": \"2020-03-19T08:47:27.102Z\", \"created\" \"2020-03-19T08:47:27.102Z\", \"user\": \"Administrator\" },\n    {\"type\" : \"info\", \"title\" : \"some title\", \"description\" : \"description\", \"x\" : 7, \"y\" : 8, \"id\": \"e43242-rewrqew-fndk423ds-jfe3xgdo\", \"modified\": \"2020-03-19T08:47:27.102Z\", \"created\" \"2020-03-19T08:47:27.102Z\", \"user\": \"Administrator\" }\n  ]\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/flows"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:51088/api/v1/flows/1/annotations' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01' -H 'Content-Type: application/json' --data '{\"type\" : \"info\", \"title\" : \"some title\", \"description\" : \"some description\", \"x\" : 23, \"y\" : 32 }'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/flows/controller.js",
    "groupTitle": "Flows"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/flows/:id/annotations/:annotId",
    "title": "Edit annotation",
    "name": "EditAnnotation",
    "group": "Flows",
    "version": "1.0.0",
    "description": "<p>This method allows you to edit a particular annotation in the flow.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          },
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-type",
            "description": "<p>application/json</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Content-type\": \"application/json\"\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Object",
            "optional": false,
            "field": "annotation",
            "description": "<p>Object containing the annotation fields that you want to update. E.g. this object may have the following structure {&quot;type&quot; : &quot;info&quot;, &quot;title&quot; : &quot;some title&quot;}</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "annotation.type",
            "description": "<p>Type of annotation (one of the following values: &quot;info&quot;, &quot;question&quot;, &quot;warning&quot;, &quot;comment&quot;)</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "annotation.title",
            "description": "<p>Title of the annotation</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "annotation.description",
            "description": "<p>Description of the annotation</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": true,
            "field": "annotation.x",
            "description": "<p>The annotation's x-coordinate on canvas</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": true,
            "field": "annotation.y",
            "description": "<p>The annotation's y-coordinate on canvas</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>URL parameter. ID of the flow.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "annotId",
            "description": "<p>URL parameter. ID of the annotation.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN</b>. If not specified, enUS will be used.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\"type\" : \"info\", \"y\" : 100}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: Annotation was successfully edited. <br> <b> false</b>: Something went wrong.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object",
            "optional": false,
            "field": "result",
            "description": "<p>Result of the operation. The format is the same as for request body with some additional fields, which were added by Switch: &quot;user&quot;(person who created the annotation), &quot;id&quot; (unique ID of the annotation), &quot;modified&quot; and &quot;created&quot; (dates in ISO format)</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true,\n  result: {\"type\" : \"info\", \"y\" : 100, \"id\": \"e43242-rewrqew-fndk423ds-jfe3xgdo\", \"modified\": \"2020-03-19T08:47:27.102Z\", \"created\" \"2020-03-19T08:47:27.102Z\", \"user\": \"Administrator\" }\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/flows"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/flows/1/annotations/431-fnuwi-43168-cms' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01' -H 'Content-Type: application/json' --data '{\"title\": \"NewTitle\", \"y\" : 53 }'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/flows/controller.js",
    "groupTitle": "Flows"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/flows",
    "title": "List",
    "name": "GetFlows",
    "group": "Flows",
    "version": "1.0.0",
    "description": "<p>This method returns flows.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "fields",
            "description": "<p>Fields that should be included in the response. Possible fields are: &quot;id&quot;, &quot;name&quot;, &quot;status&quot;, &quot;groups&quot;, &quot;stages&quot;, &quot;customJobFields&quot; and &quot;annotations&quot;. If no fields are specified explicitly, all fields are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "ids",
            "description": "<p>Optional parameter that points to flow IDs. If specified, only flows with the corresponding IDs will be returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN</b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Flow ID. Note: When the flow status is &quot;removed&quot;, the flow ID is 0.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "name",
            "description": "<p>Name of the flow.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "description",
            "description": "<p>Description of the flow.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "status",
            "description": "<p>Status of the flow. Possible values: &quot;running&quot;, &quot;stopped&quot;, &quot;scheduled&quot;, &quot;removed&quot;.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": true,
            "field": "groups",
            "description": "<p>The list of groups the flow belongs to (starting from the direct parent). Each group is an object with two fields: &quot;id&quot; and &quot;name&quot;. For example: {&quot;id&quot;:&quot;1&quot;,&quot;name&quot;:&quot;Test&quot;}.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "groups.id",
            "description": "<p>Unique ID of the group.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "groups.name",
            "description": "<p>Name of the group.</p>"
          },
          {
            "group": "Success 200",
            "type": "String[]",
            "optional": true,
            "field": "stages",
            "description": "<p>The list of stages in the flow.</p>"
          },
          {
            "group": "Success 200",
            "type": "String[]",
            "optional": true,
            "field": "customJobFields",
            "description": "<p>The list of job custom fields set in the flow.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": true,
            "field": "annotations",
            "description": "<p>The list of flow annotations.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "annotations.id",
            "description": "<p>Annotation's ID.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "annotations.type",
            "description": "<p>Type of annotation. Possible values: &quot;info&quot;, &quot;question&quot;, &quot;warning&quot;, &quot;comment&quot;.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "annotations.title",
            "description": "<p>Title of the annotation.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "annotations.description",
            "description": "<p>Description of the annotation.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "annotations.x",
            "description": "<p>The annotation's x-coordinate on canvas.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "annotations.y",
            "description": "<p>The annotation's y-coordinate on canvas.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "annotations.user",
            "description": "<p>The user who created or modified the annotation.</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": false,
            "field": "annotations.created",
            "description": "<p>Creation date of the annotation.</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": false,
            "field": "annotations.modified",
            "description": "<p>Modification date of the annotation.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"id\":3,\n  \"stages\": [\n     \"Problem jobs\"\n   ],\n  \"name\": \"New flow\",\n  \"status\":\"running\",\n  \"groups\":\"[{\"id\":\"1\",\"name\":\"Test\"}],\n  \"annotations\": \"[{\"type\" : \"info\", \"title\" : \"JobSubmitter\", \"description\" : \"Allow user submit jobs\", \"x\" : 3, \"y\" : 44, \"user\": \"Administrator\", \"modified\": \"2020-03-12T09:01:35.376Z\", \"created\": \"2020-03-12T09:01:35.376Z\" },\n                   {\"type\" : \"comment\", \"title\" : \"Element\", \"description\" : \"Some comment here\", \"x\" : 12, \"y\" : 23, \"user\": \"Administrator\", \"modified\": \"2020-03-12T09:01:35.376Z\", \"created\": \"2020-03-12T09:01:35.376Z\" }\n                 ]\"\n},",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error <br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error { \"status\": false, \"error\": 'Cannot get flows.' }",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/flows"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl 'http://127.0.0.1:51088/api/v1/flows' -H 'Authorization: Bearer c5b6583634cf25e8fbfe7fbe972dc2fc883db953'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/flows/controller.js",
    "groupTitle": "Flows"
  },
  {
    "type": "delete",
    "url": "http://127.0.0.1:51088/api/v1/flows/:id/annotations/:annotId",
    "title": "Remove annotations",
    "name": "RemoveAnnotations",
    "group": "Flows",
    "version": "1.0.0",
    "description": "<p>This method allows you to remove annotations from a particular flow.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>URL parameter. ID of the flow.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "annotId",
            "description": "<p>URL parameter. ID of the flow annotation, that should be removed. If not specified, all annotations will be removed from the flow.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN</b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: Annotations were successfully removed. <br> <b> false</b>: Something went wrong.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/flows"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X DELETE 'http://127.0.0.1:51088/api/v1/flows/1/annotations/431-fnuwi-43168-cms' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/flows/controller.js",
    "groupTitle": "Flows"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/flows/:id?action=start",
    "title": "Start",
    "name": "StartFlow",
    "group": "Flows",
    "version": "1.0.0",
    "description": "<p>This method allows you to start a particular flow based on its flow ID. The Switch Web Services wait till the flow is completely started.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Flow ID. It can be retrieved using the List flows call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN</b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: The flow was successfully started. <br> <b> false</b>: Something went wrong while starting the flow.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "flowStatus",
            "description": "<p>new status of the flow (either 'scheduled' or 'running')</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true,\n  flowStatus: \"running\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Flow with ID 655 does not exist.'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/flows?action=start"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/flows/1?action=start&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/flows/controller.js",
    "groupTitle": "Flows"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/flows/:id?action=stop",
    "title": "Stop",
    "name": "StopFlow",
    "group": "Flows",
    "version": "1.0.0",
    "description": "<p>This method allows you to stop a particular flow based on its flow ID. The Switch Web Services wait till the flow is completely stopped.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Flow ID. The flow ID can be retrieved using the List flows call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN</b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: The flow was successfully stopped. <br> <b> false</b>: Something went wrong while stopping the flow.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Flow with ID 655 does not exist.'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/flows?action=stop"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/flows/1?action=stop&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/flows/controller.js",
    "groupTitle": "Flows"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/job/metadata",
    "title": "Metadata",
    "name": "GetJobMetadata",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method returns a job's metadata definition.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "ids",
            "description": "<p>Comma separated list of job IDs for which metadata should be returned. Job IDs can be retrieved using the List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "readonly",
            "description": "<p>Allows the user to request either read-only metadata fields or editable fields. If not specified, all metadata fields are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Status of the executed operation<br> <b>true</b>: Metadata was successfully calculated. <br> <b>false</b>: Something went wrong during metadata calculation.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error string in case the status is false.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": false,
            "field": "metadata",
            "description": "<p>Set of metadata fields for all requested jobs. Keys of this set represent job IDs. Values are arrays of calculated metadata instances for the job. For more details, refer to the example below.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.id",
            "description": "<p>Unique ID of the metadata field.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "metadata.displayField",
            "description": "<p>Tells the client whether it should display the field to the user or not.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.type",
            "description": "<p>Type of the metadata field. Possible values: enum, string, password, number, bool, time, date.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.name",
            "description": "<p>Name of the metadata field.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.value",
            "description": "<p>Default value of the metadata field.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.description",
            "description": "<p>Description for the metadata field.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "metadata.valueIsRequired",
            "description": "<p>True if the metadata field is required and false otherwise.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.format",
            "description": "<p>Regular expression which determines the format of the value for the metadata field. Should be used to validate the value.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "metadata.readOnly",
            "description": "<p>If true, the user should not have the possibility to edit the (default) value.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "metadata.rememberLastValue",
            "description": "<p>Tells the client whether it should remember the last value entered by the user to show it as default for the next job submission.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "metadata.ignoreDuplicates",
            "description": "<p>Tells the client whether it should show the duplicated values (applicable only if type is enum).</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.dependency",
            "description": "<p>ID of a parent metadata field on which the current field depends.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.dependencyCondition",
            "description": "<p>String which represents the method to check dependency. Possible values are: &quot;Equals&quot;, &quot;Not-equals&quot;, &quot;Contains&quot;, &quot;Does not contain&quot;, &quot;Starts with&quot;, &quot;Does not start with&quot;, &quot;Matches&quot;, &quot;Does not match&quot;.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "metadata.dependencyValue",
            "description": "<p>Value of the parent metadata field, which allows the client to show the current metadata field to the user. So dependencyCondition and dependencyValue are applied on dependency (parent field) to check whether or not the current field should be shown.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"status\": true,\n  \"metadata\": {\n    \"5851177ec172b7b936a4754b\": [\n      {\n        \"dependency\":\"\",\n        \"dependencyCondition\":\"\",\n        \"dependencyValue\":\"\",\n        \"displayField\":true,\n        \"format\":\"\",\n        \"id\":\"cpMF_1\",\n        \"ignoreDuplicates\":false,\n        \"name\":\"Metadata Field 1\",\n        \"readOnly\":false,\n        \"rememberLastValue\":false,\n        \"description\":\"\",\n        \"type\":\"string\",\n        \"value\":\"\",\n        \"valueIsRequired\":false\n      },{\n        \"dependency\": \"spMF_1\",\n        \"dependencyCondition\": \"Equals\",\n        \"dependencyValue\": \"46\"\n        \"displayField\":true,\n        \"format\":\"10.*\",\n        \"id\":\"spMF_1\",\n        \"ignoreDuplicates\":false,\n        \"name\":\"Metadata Field 1\",\n        \"readOnly\":true,\n        \"rememberLastValue\":true,\n        \"description\":\"\",\n        \"type\":\"string\",\n        \"value\":\"10\",\n        \"valueIsRequired\":false\n      }\n    ]\n  }\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 401": [
          {
            "group": "Error 401",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 403": [
          {
            "group": "Error 403",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false</p>"
          },
          {
            "group": "Error 403",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n   \"status\": false,\n   \"error\": 'Cannot calculate requested job(s) metadata.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job/metadata"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/job/metadata?ids=5851177ec172b7b936a4754b&readonly=true&lang=deDE' -H 'Authorization: Bearer 901cc715a46298cb8731b9e3f85d14e4d506f8c7'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/job/:id",
    "title": "Download",
    "name": "JobDownload",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method returns a link to download the job.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job ID. It can be retrieved using the List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "200": [
          {
            "group": "200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          },
          {
            "group": "200",
            "type": "String",
            "optional": false,
            "field": "data",
            "description": "<p>A link to download the job. This link is valid only for the current user and the current session. In case the job is a job folder, this link points to the archived job folder.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"status\": true,\n  \"data\": \"http://127.0.0.1:51088/job/7b226964223a22353835313137376563313732623762393336613437353462222c22756964223a2262373637633731666162613966663762353465353330613063626639393664646233313531333164227d\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 401": [
          {
            "group": "Error 401",
            "type": "String",
            "optional": false,
            "field": "Unauthorized",
            "description": ""
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": 'Job does not exist or access denied.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job"
      }
    ],
    "examples": [
      {
        "title": "Example usage step 1:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/job/5851177ec172b7b936a4754b?lang=deDE' -H 'Authorization: Bearer 901cc715a46298cb8731b9e3f85d14e4d506f8c7'",
        "type": "curl"
      },
      {
        "title": "Example usage step 2:",
        "content": "curl -X GET 'http://127.0.0.1:51088/job/7b226964223a22353835376565363635366662646365333535636166363338222c22756964223a2230383362356264653435383031366530386339303033333934633638376266323163353835393330227d'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/job/report/:id",
    "title": "Report",
    "name": "JobReportDownload",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method returns a link to download the report associated with the job.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job ID of the job for which you want to download the report. It can be retrieved using the List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "200": [
          {
            "group": "200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          },
          {
            "group": "200",
            "type": "String",
            "optional": false,
            "field": "data",
            "description": "<p>A link to download the report associated with the job. This link is valid only for the current user and the current session.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"status\": true,\n  \"data\": \"http://127.0.0.1:51088/job/report/7b226964223a22353864306562336166383166353739643630306463323835222c22756964223a2237633731326633343337313336303439373865376435626161333237356463646366613434656635227d\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 401": [
          {
            "group": "Error 401",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 403": [
          {
            "group": "Error 403",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false</p>"
          },
          {
            "group": "Error 403",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'Report viewing is not enabled for the checkpoint'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": 'Job does not exist or access denied.'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": 'Report file does not exist.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job/report"
      }
    ],
    "examples": [
      {
        "title": "Example usage step 1:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/job/report/5851177ec172b7b936a4754b?lang=deDE' -H 'Authorization: Bearer 901cc715a46298cb8731b9e3f85d14e4d506f8c7'",
        "type": "curl"
      },
      {
        "title": "Example usage step 2:",
        "content": "curl -X GET 'http://127.0.0.1:51088/job/report/7b226964223a22353864306562336166383166353739643630306463323835222c22756964223a2237633731326633343337313336303439373865376435626161333237356463646366613434656635227d'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/job/:id?action=lock",
    "title": "Lock",
    "name": "LockJob",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method locks the specified job so that it cannot be routed or replaced by any user other than an administrator or a group manager of the group the current user belongs to.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job ID. It can be retrieved using the List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: The job was successfully locked. <br> <b> false</b>: Something went wrong while locking the job.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'The job is locked by another user'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Job does not exist or access denied'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/job/5851177ec172b7b936a4754b?action=lock&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:51088/api/v1/job",
    "title": "Submit",
    "name": "PostJob",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method submits a job to a Submit point.</p> <p>The content type of the HTTP request body must be &quot;multipart/form-data&quot;. This type is defined in RFC 7578.</p> <p>A multipart/form-data body contains a series of parts separated by a boundary. The Switch Web Services expect that each part contains a piece of information about the job being submitted: job name, content, metadata etc.</p> <p>The boundary is supplied as a &quot;boundary&quot; parameter to the multipart/form-data type. This means the HTTP request must include a Content-Type HTTP header field of the following form: <i>Content-Type: multipart/form-data; boundary=<boundary delimiter></i> For example: <i>Content-Type: multipart/form-data; boundary=----WebKitFormBoundarysHGON0xlKXKSfDuN</i> The boundary delimiter must not appear inside any of the encapsulated parts. The boundaries are used by the Switch Web Services to extract the individual parts of the HTTP request body containing specific information about the job. Each part must contain a Content-Disposition header field where the disposition type is &quot;form-data&quot;. The Content-Disposition header field must also contain an additional parameter &quot;name&quot;; the value of the &quot;name&quot; parameter indicates which piece of information about the job this part represents. The possible values for the name parameter are : flowId, objectId, metadata etc. The table &quot;Parameter&quot; below describes all possible values. For example, the header for the part describing the ID of the flow looks like the following:<br> <i>Content-Disposition: form-data; name=&quot;flowId&quot;</i> The request might include one or more files. If more than one file is included, each file is represented by two related parts in the multipart/form-data body which define the file path and the file content. The Content-Disposition headers for these parts look like the following:<br> <i>Content-Disposition: form-data; name=&quot;file[0][path]&quot;</i><br> <i>Content-Disposition: form-data; name=&quot;file[0][file]&quot;</i><br> If only one file is included, it is represented only by the file content. For example, when submitting the job test.txt, the request body will include the following parts:</p> <pre>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][file]\"<br>   Content-Type: text/plain<br><br>     |test.txt content| </pre> <p>Submitting folders is possible by specifying relative paths for files in the folder. For example, to upload &quot;folder1&quot; with two files &quot;text1.txt&quot; and &quot;text2.txt&quot; the following data should be included in the request body:</p> <pre>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][path]\"<br><br>     folder1/test1.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][file]\"<br>   Content-Type: text/plain<br><br>     |test1.txt content|<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[1][path]\"<br><br>     folder1/test2.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[1][file]\"<br>   Content-Type: text/plain<br><br>     |test2.txt content| </pre> <p>The example of the complete request body:</p> <pre>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"flowId\"<br><br>     2<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"objectId\"<br><br>     3<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"jobName\"<br><br>     JobSample<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"metadata\"<br><br>     [{<br>        \"id\": \"spMF_1\",<br>        \"name\": \"Password\",<br>        \"value\": \"foo\"<br>     }]<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"origin\"<br><br>     /Volumes/HD/Users/Admin/MyJob.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"created\"<br><br>     2016-11-27T20:41:22.535Z<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"modified\"<br><br>     2016-11-28T20:41:22.535Z<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][path]\"<br><br>     folder1/example1.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][file]\"<br>   Content-Type: text/plain<br><br>     |example1.txt content|<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[1][path]\"<br><br>     folder1/example2.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[1][file]\"<br>   Content-Type: text/plain<br><br>     |example2.txt content|<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[2][path]\"<br><br>     folder1/folder2/example3.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[2][file]\"<br>   Content-Type: text/plain<br><br>     |example3.txt content|<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN </pre>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          },
          {
            "group": "Header",
            "type": "Number",
            "optional": false,
            "field": "Content-Length",
            "description": "<p>Content length</p>"
          },
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-Type",
            "description": "<p>Should be multipart/form-data with boundary</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n  \"Content-Length\": 1373\n  \"Content-Type\": \"multipart/form-data; boundary=----WebKitFormBoundaryrGXxz3Kn1K5R3kAB\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "flowId",
            "description": "<p>ID of the flow where the job should be submitted. It can be retrieved using the List submit points call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "objectId",
            "description": "<p>ID of the Submit point where the job should be submitted. It can be retrieved using the List submit points call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "filePath",
            "description": "<p>file|index||path| The relative path to the file in the job folder. The path is used by the Switch Web Services to create the folder structure on disk. The job will afterwards be renamed as specified by the jobName parameter. This parameter is only used/applicable in case of job folders, not in case of single files.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "fileContent",
            "description": "<p>file|index||file| Binary file data.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "jobName",
            "description": "<p>Job name. This can be a different name than the initial file name (with appropriate extension if it is not a folder): the job will be renamed on the server according to this parameter.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "origin",
            "description": "<p>The job's original path on the client system. If not specified, the path to the job in the Submit point backing folder is used.</p>"
          },
          {
            "group": "Parameter",
            "type": "Date",
            "optional": true,
            "field": "created",
            "description": "<p>Job creation date and time in ISO format.</p>"
          },
          {
            "group": "Parameter",
            "type": "Date",
            "optional": true,
            "field": "modified",
            "description": "<p>Job modification date and time in ISO format.</p>"
          },
          {
            "group": "Parameter",
            "type": "Object[]",
            "optional": true,
            "field": "metadata",
            "description": "<p>Array of job metadata in Switch metadata format.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.id",
            "description": "<p>Unique ID of the metadata field. It can be retrieved using the List submit points call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.name",
            "description": "<p>Name of the metadata field.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.value",
            "description": "<p>Value of the metadata field.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "200": [
          {
            "group": "200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          },
          {
            "group": "200",
            "type": "String",
            "optional": false,
            "field": "jobId",
            "description": "<p>ID that can be used to refer to this job in all calls of the Switch Client module API.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  \"status\": true,\n  \"jobId\": \"604f4d59822f58c7c52dff04\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 500": [
          {
            "group": "Error 500",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false</p>"
          },
          {
            "group": "Error 500",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message</p>"
          }
        ],
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot submit a job.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job"
      }
    ],
    "examples": [
      {
        "title": "Curl example with manual setting multipart form data:",
        "content": "curl -X POST 'http://127.0.0.1:51088/api/v1/job?lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef' -H 'Content-Type: multipart/form-data; boundary=----WebKitFormBoundaryLXxoX58ScYq3mwdt' --data-binary $'------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"file\"\\r\\nContent-Type: text/plain\\r\\n\\r\\nsampleContent\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"flowId\"\\r\\n\\r\\n1\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"objectId\"\\r\\n\\r\\n2\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"filePath\"\\r\\n\\r\\nsample.txt\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"jobName\"\\r\\n\\r\\nsample.txt\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"modified\"\\r\\n\\r\\n2016-09-30T08:18:50.000Z\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"metadata\"\\r\\n\\r\\n[]\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt--\\r\\n'",
        "type": "json"
      },
      {
        "title": "Curl example with auto setting multipart form data:",
        "content": "curl -X POST 'http://127.0.0.1:51088/api/v1/job?lang=deDE' -H 'Authorization: Bearer 80af6a1ceb108ea6193c50745616273d5543146b' -F \"file[0][file]=@sample.txt\" -F \"flowId=3\" -F \"objectId=2\" -F \"file[0][path]=sample.txt\" -F \"jobName=sample.txt\"",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/job/:id?action=replace",
    "title": "Replace",
    "name": "ReplaceJob",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method replaces a specified job in a Checkpoint. The HTTP POST method is also supported for this call.<br> The content type of the HTTP request body must be &quot;multipart/form-data&quot;. This type is defined in RFC 7578. Fore more information about the body format, refer to the <a href=\"#api-Job-PostJob\">SubmitJob</a> specification.<br></p> <p>The Replace Job request can include one or more files. If more than one file is included, each file is represented by two related parts in the multipart/form-data body which define the file path and the file content. The Content-Disposition headers for these parts must have the following format:<br> <i>Content-Disposition: form-data; name=&quot;file[0][path]&quot;</i><br> <i>Content-Disposition: form-data; name=&quot;file[0][file]&quot;; filename=&quot;test.txt&quot;</i><br> If only one file is included, it is represented only by the file content. We allow to change the extension of the job on replace by passing on the new extension to the &quot;filename&quot; parameter of the Content-Disposition header field for the part that represents the content of the file.<br> For example, to replace the job <i>test.pdf</i> with <i>test.txt</i>, the request body must include the following parts:</p> <pre>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"updated\"<br><br>     2017-03-17T14:24:13<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][file]\"; filename=\"test.txt\"<br>   Content-Type: text/plain<br><br>     |test.txt content| </pre> <p>To replace a job with a folder, you must use relative file paths.<br>For example, to replace a job with <i>folder1</i>, which contains two files <i>text1.txt</i> and <i>text2.txt</i>, the request body must include the following parts:</p> <pre>    ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"updated\"<br><br>     2017-03-17T14:24:13<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][path]\"<br><br>     folder1/test1.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[0][file]\"<br>   Content-Type: text/plain<br><br>     |test1.txt content|<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[1][path]\"<br><br>     folder1/test2.txt<br>   ------WebKitFormBoundarysHGON0xlKXKSfDuN<br>   Content-Disposition: form-data; name=\"file[1][file]\"<br>   Content-Type: text/plain<br><br>     |test2.txt content| </pre>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          },
          {
            "group": "Header",
            "type": "Number",
            "optional": false,
            "field": "Content-Length",
            "description": "<p>Content length</p>"
          },
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-Type",
            "description": "<p>Should be multipart/form-data with boundary</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n  \"Content-Length\": 1373\n  \"Content-Type\": \"multipart/form-data; boundary=----WebKitFormBoundaryrGXxz3Kn1K5R3kAB\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job ID. It can be retrieved using the List jobs call. Unlike all other parameters, the job ID is passed via a URL.</p>"
          },
          {
            "group": "Parameter",
            "type": "Date",
            "optional": true,
            "field": "updated",
            "description": "<p>Date in ISO format (e.g. 2017-03-17T14:24:13). This field is used to check the job relevance before replacing. Pass the value of the 'updated' field for the job being replaced:<br> If the value is different from the actual value, the operation is cancelled and an error is returned.<br> If no value is specified, replacing proceeds without job relevance check.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "filePath",
            "description": "<p>file|index||path| The relative path to the file in the job folder. It is used by the Switch Web Services to create the folder structure on disk and it does not influence the original job name. This parameter is only used/applicable in case of job folders, not in case of single files. In case of a job folder, the filePath parameter will be used as relative path for the file in the job folder but the original job folder name will be preserved.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "fileContent",
            "description": "<p>file|index||file| Binary file data. If a job file is replaced a &quot;filename&quot; parameter should be supplied as well as part of the fileContent. The extension of the &quot;filename&quot; parameter will be used as extension for the replaced file.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag. <br> <b> true</b>: The job was successfully replaced. <br> <b> false</b>: Something went wrong during replacing.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden (<b>NOTE:</b> The Checkpoint property <i>Allow replacing or editing job</i> should be set to true to allow users to replace jobs.)</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'Job information has changed since you got it last time'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Job does not exist or access denied'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job"
      }
    ],
    "examples": [
      {
        "title": "Curl example with manual setting multipart form data:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/job/5851177ec172b7b936a4754b?action=replace&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef' -H 'Content-Type: multipart/form-data; boundary=----WebKitFormBoundaryLXxoX58ScYq3mwdt' --data-binary $'------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"file[0][file]\"; filename=\"sample.txt\"\\r\\nContent-Type: text/plain\\r\\n\\r\\nsampleContent\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt\\r\\nContent-Disposition: form-data; name=\"file[0][path]\"\\r\\n\\r\\nsample.txt\\r\\n------WebKitFormBoundaryLXxoX58ScYq3mwdt--\\r\\n'",
        "type": "json"
      },
      {
        "title": "Curl example with auto setting multipart form data:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/job/59c8cf95eead70fa2363c880?action=replace&lang=deDE' -H 'Authorization: Bearer 80af6a1ceb108ea6193c50745616273d5543146b' -F \"file[0][file]=@sample.txt\" -F \"file[0][path]=sample.txt\" -F \"fileName=sample.txt\"",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/job/:id?action=route",
    "title": "Route",
    "name": "RouteJob",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method routes the specified job to the specified folder(s).</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job ID. It can be retrieved using the List jobs call. Unlike all other parameters, the job ID is passed via a URL.</p>"
          },
          {
            "group": "Parameter",
            "type": "Date",
            "optional": true,
            "field": "updated",
            "description": "<p>Used to check the job relevance before routing. Pass the value of the 'updated' field for the job being routed. If it conflicts with the actual value, the routing operation will be cancelled and an error will be returned. If not specified, routing proceeds without job relevance check.</p>"
          },
          {
            "group": "Parameter",
            "type": "Array",
            "optional": false,
            "field": "connections",
            "description": "<p>List of connection IDs to route the job to.</p>"
          },
          {
            "group": "Parameter",
            "type": "Object[]",
            "optional": true,
            "field": "metadata",
            "description": "<p>Array of job metadata in the Switch metadata format.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.id",
            "description": "<p>Unique ID of the metadata field. It can be retrieved using the Job metadata call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.name",
            "description": "<p>Name of the metadata field.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.value",
            "description": "<p>Value of the metadata field.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"connections\" : [\"5\",\"7\",\"10\"],\n  \"updated\" : \"2016-11-25T16:58:49\",\n  \"metadata\":[{\n    \"id\": \"cpMF_1\",\n    \"name\": \"Metadata Field 1\",\n    \"value\": \"value 1\"\n  }, {\n    \"id\": \"cpMF_2\",\n    \"name\": \"Metadata Field 2\",\n    \"value\": \"value 2\"\n  }]\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Bool flag. <br> <b> true</b>: The job was successfully routed. <br> <b> false</b>: Something went wrong during routing.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  \"status\": true,\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'Multiple output is not allowed for the checkpoint'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal server error\n{\n  \"status\": false,\n  \"error\": 'Internal server error.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/job/5851177ec172b7b936a4754b?action=route&lang=deDE' -H 'Authorization: Bearer 41db197826c363cf89a6f34cce47ec666378cd0b' --data 'connections=%5B%5D&metadata=%5B%5D'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/job/:id?action=unlock",
    "title": "Unlock",
    "name": "UnlockJob",
    "group": "Job",
    "version": "1.0.0",
    "description": "<p>This method unlocks the locked job. It's also possible to only check the user's permission to unlock the job.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job ID. It can be retrieved using the List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "check",
            "description": "<p>Used to check whether or not the user is able to unlock the job.<br> If set to true, the user's ability to unlock the job is checked. Note that it's a check only, this parameter does not unlock the job!<br>If set to false (default value), the user's permissions are checked, and the job is unlocked if the user has the right to do so.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: The job was successfully unlocked. <br> <b> false</b>: Something went wrong while unlocking the job.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'The job is locked by another user'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Job does not exist or access denied'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/job"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/job/5851177ec172b7b936a4754b?action=unlock&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/job/controller.js",
    "groupTitle": "Job"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/processingjob/:processingId?action=rush",
    "title": "Rush",
    "name": "RushJob",
    "group": "Job_Processing",
    "version": "1.0.0",
    "description": "<p>This method changes the priority of a job, to make sure that the job is processed before any other job in Switch. It puts the job at the end of the rush job queue. When applied to an existing rush job, it will put the rush job back at the end of the rush job queue. The rank of the other jobs is not changed in this case. A rush call is only allowed for users who have the &quot;Rush Job&quot; permission to rush jobs.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "processingId",
            "description": "<p>Job ID, which can be retrieved using the List jobs call. Note that this is the processing ID (e.g. 000A2) of the job, not the job client ID.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: New rank was successfully assigned to job. <br> <b> false</b>: Something went wrong while assigning the rank.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'Operation forbidden'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Job not found'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/processingjob"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/processingjob/00001?action=rush&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/processingJob/controller.js",
    "groupTitle": "Job_Processing"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/processingjob/:processingId?action=unrush",
    "title": "Unrush",
    "name": "UnrushJob",
    "group": "Job_Processing",
    "version": "1.0.0",
    "description": "<p>This method removes any priorities from a job. This call is only allowed for users who have the permission to rush jobs.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "processingId",
            "description": "<p>Job ID, which can be retrieved using the List jobs call. Note that this is the processing ID (e.g. 000A2) of the job, not the job client ID.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Boolean flag <br> <b> true</b>: Rank was successfully removed from the job. <br> <b> false</b>: Something went wrong while removing the rank.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message in case the status is false.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden</p>"
          },
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error<br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  status: false,\n  error: 'Operation forbidden'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  status: false,\n  error: 'Job not found'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Internal server error'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/processingjob"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/processingjob/00001?action=unrush&lang=deDE' -H 'Authorization: Bearer e6587350eac67d8f96d0757a98fd02ab316707ef'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/processingJob/controller.js",
    "groupTitle": "Job_Processing"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:51088/api/v1/jobFilters",
    "title": "Add",
    "name": "AddJobFilter",
    "group": "Job_filters",
    "version": "1.0.0",
    "description": "<p>This method allows the user to save a job filter in Switch, so it can be retrieved and applied later on.</p> <p>The Switch job filter query is in JSON format, which allows the user to set up complex job filters which can be used in Get jobs and Get GraphQL processing jobs requests. For the whole list of job filter fields, see the result returned by the Get jobs request.</p> <p>String operators:</p>   <pre> {\"and\":[{\"name\":{\"contains\":\"tx\"}}]} </pre>   <pre> {\"and\":[{\"name\":{\"does_not_contain\":\"tx\"}}]} </pre>   <pre> {\"and\":[{\"type\":{\"is\":\"txt\"}}]} </pre>   <pre> {\"and\":[{\"type\":{\"is_not\":\"txt\"}}]} </pre>   <pre> {\"and\":[{\"name\":{\"starts_with\":\"folder\"}}]} </pre>   <pre> {\"and\":[{\"name\":{\"ends_with\":\"txt\"}}]} </pre> <p><b>Note:</b> Unlike other string fields, the (job) id field only supports the &quot;is&quot; operator.</p> <p>Number operators:</p>   <pre> {\"and\":[{\"size\":{\"num_is\":\"14\"}}]} </pre>   <pre> {\"and\":[{\"size\":{\"num_is_not\":\"12\"}}]} </pre>   <pre> {\"and\":[{\"size\":{\"is_greater_than\":\"12\"}}]} </pre>   <pre> {\"and\":[{\"size\":{\"is_less_than\":\"16\"}}]} </pre>   <pre> {\"and\":[{\"size\":{\"is_in_the_range\":\"12 16\"}}]} </pre> <p><b>Note:</b> All operators for size accept size metrics: MB, KB, bytes. Values without metric are considered as bytes.</p>   <pre> {\"and\":[{\"size\":{\"num_is\":\"14;bytes\"}}]} </pre>   <pre> {\"and\":[{\"size\":{\"is_less_than\":\"14;MB\"}}]} </pre>   <pre> {\"and\":[{\"size\":{\"is_in_the_range\":\"10;bytes 10;KB\"}}]} </pre> <p>Date operators:</p>   <pre> {\"and\":[{\"initiated\":{\"date_is\":\"2016-09-30T11:18:50\"}}]} </pre>   <pre> {\"and\":[{\"initiated\":{\"date_is_not\":\"2016-09-30T11:18:50\"}}]} </pre>   <pre> {\"and\":[{\"initiated\":{\"is_after\":\"2016-09-30T11:18:50\"}}]} </pre>   <pre> {\"and\":[{\"initiated\":{\"is_before\":\"2016-09-30T11:18:50\"}}]} </pre>   <pre> {\"and\":[{\"initiated\":{\"is_in_the_last\":\"5;days\"}}]} </pre>   <pre> {\"and\":[{\"initiated\":{\"is_not_in_the_last\":\"5;days\"}}]} </pre>   <pre> {\"and\":[{\"initiated\":{\"date_is_in_the_range\":\"2016-09-30T11:18:50 2016-09-30T11:18:50\"}}]} </pre> <p><b>Note:</b> All operators for the fields with date type accept metrics: months, weeks, days for values is_in_the_last and is_not_in_the_last. Values without metric are considered as days.</p> <p>Boolean operators:</p>   <pre> {\"and\":[{\"hasMetadata\":{\"bool_is\":\"true\"}}]} </pre>   <pre> {\"and\":[{\"hasMetadata\":{\"bool_is_not\":\"false\"}}]} </pre> <p><em>is_current_user</em> operator is the special operator to request jobs that belong to the current user. <b>Note:</b> It can be applied to userName field only.</p>   <pre> {\"and\":[{\"userName\":{\"is_current_user\":\"true\"}}]} </pre> <p>Nested job filter example:</p>   <pre> {\"and\":[{\"userName\":{\"is\":\"Administrator\"}},{\"type\":{\"is\":\"txt\"}},{\"and\":[{\"size\":{\"num_is\":\"14;bytes\"}},{\"and\":[{\"submittedTo\":{\"starts_with\":\"Submit point\"}}]},{\"or\":[{\"family\":{\"contains\":\"0\"}},{\"files\":{\"num_is\":\"1\"}}]}]}]} </pre>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "name",
            "description": "<p>Job filter name</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "query",
            "description": "<p>Job filter query in the format described above.</p>"
          },
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "visibility",
            "description": "<p>Indicates whether the job filter is visible for the user or not, true/false. Default is true.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"name\" : \"my after filter\",\n  \"query\" : \"{\\\"and\\\":[{\\\"initiated\\\":{\\\"is_after\\\":\\\"2016-09-30T11:18:50\\\"}}]}\",\n  \"visibility\": true\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Automatically generated ID of the just added job filter</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  \"id\": \"9d0b43e0-7696-11e3-964b-123456789012\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot add job filter.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/jobFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:51088/api/v1/jobFilters' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01' -H 'Content-Type: application/json' --data '{\"name\": \"filterName\", \"query\": {\"and\":[{\"type\":{\"is\":\"txt\"}}]}, \"visibility\": true}'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobfilters/controller.js",
    "groupTitle": "Job_filters"
  },
  {
    "type": "delete",
    "url": "http://127.0.0.1:51088/api/v1/jobFilters",
    "title": "Remove",
    "name": "DeleteJobFilter",
    "group": "Job_filters",
    "version": "1.0.0",
    "description": "<p>This method removes a job filter. Note that it is not allowed to remove predefined job filters.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "ids",
            "description": "<p>Comma separated list of job filter IDs. If not specified, all job filters for the current user will be removed.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success",
          "content": "HTTP/1.1 200\n{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found. Returned when the requested filter is not found or when the user tries to remove a predefined job filter.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": \"Cannot remove the job filters: Error: Cannot find filters.\"\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot remove job filters.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/jobFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X DELETE 'http://127.0.0.1:51088/api/v1/jobFilters?ids=3363eb10-bafc-11e6-83b8-8d274e5d1b3b' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobfilters/controller.js",
    "groupTitle": "Job_filters"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/jobFilters/:id",
    "title": "Edit",
    "name": "EditJobFilter",
    "group": "Job_filters",
    "version": "1.0.0",
    "description": "<p>This method allows the user to edit a job filter in Switch. Note that it is not allowed to edit the predefined job filters.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job filter ID. Unlike all other parameters, the job filter ID is passed via a URL.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "name",
            "description": "<p>Job filter name. Note: Only the owner of the job filter can edit this field. If a user who is not the owner tries to change it, he will get an error.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "query",
            "description": "<p>Job filter query in the format described above. Note: Only the owner of the job filter can edit this field. If a user who is not the owner tries to change it, he will get an error.</p>"
          },
          {
            "group": "Parameter",
            "type": "Object",
            "optional": true,
            "field": "visibility",
            "description": "<p>Indicates whether the job filter is visible for the user or not, true/false.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"name\" : \"my after filter\",\n  \"query\" : \"{\\\"and\\\":[{\\\"initiated\\\":{\\\"is_after\\\":\\\"2016-09-30T11:18:50\\\"}}]}\",\n  \"visibility\": true\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success",
          "content": "HTTP/1.1 200\n{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden. Returned when the user is not allowed to change the job filter (typically a predefined job filter).</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found. Returned when the requested job filter is not found.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  \"status\": false,\n  \"error\": \"Cannot edit the job filter. Error: The requested action is not allowed.\"\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": \"Cannot edit the job filter. Error: The filter does not exist.\"\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot edit the job filter.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/jobFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/jobFilters/3363eb10-bafc-11e6-83b8-8d274e5d1b3b' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01' -H 'Content-Type: application/json' --data '{\"name\": \"filterName\", \"query\": {\"and\":[{\"type\":{\"is\":\"txt\"}}]}, \"visibility\": true}'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobfilters/controller.js",
    "groupTitle": "Job_filters"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/jobFilters",
    "title": "List",
    "name": "GetJobFilter",
    "group": "Job_filters",
    "version": "1.0.0",
    "description": "<p>This call returns the list of job filters for the current user.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "ids",
            "description": "<p>Comma separated list of job filter IDs. If not specified, all job filters for the current user will be returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "fields",
            "description": "<p>List of fields to return. Supported fields: <b>id, name, query, shared, visibility, user, type</b>. If not specified, all fields are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          },
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "includePredefined",
            "description": "<p>Indicates whether or not the response should include the predefined job filters (type=predefinedCheckJobs). If not set, the predefined job  filters will not be returned.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "id",
            "description": "<p>Job filter ID automatically assigned by Switch.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "name",
            "description": "<p>Job filter name. Predefined job filter names: <b>&quot;alertJobs&quot;, &quot;completedJobs&quot;, &quot;myJobs&quot;, &quot;activeJobs&quot;</b>. <br> There are 4 predefined job filters:</p> <ul>   <li>alertJobs - Returns jobs that are in Alert status and need to be reviewed.</li>   <li>completedJobs - Returns jobs that are in Completed status. These jobs have been processed completely.</li>   <li>myJobs - Returns jobs that belong to the user that is currently logged in.</li>   <li>activeJobs - Returns jobs that are currently being processed by Switch (status is Processing).</li> </ul>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "query",
            "description": "<p>Job filter query <b>as encoded URI</b> in the format described in Job filters - Add section.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object",
            "optional": true,
            "field": "shared",
            "description": "<p>The object shows with whom the job filters are shared. Accessible only for the owner of the job filter.</p>"
          },
          {
            "group": "Success 200",
            "type": "Array",
            "optional": true,
            "field": "shared.groups",
            "description": "<p>The list of user groups the job filter is shared with.</p>"
          },
          {
            "group": "Success 200",
            "type": "Array",
            "optional": true,
            "field": "shared.users",
            "description": "<p>The list of users the job filter is shared with.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "visibility",
            "description": "<p>Indicates whether the job filter is visible for the user or not, true/false.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "user",
            "description": "<p>The creator of the job filter. Empty if the job filter is predefined.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "type",
            "description": "<p>The type of job filter. Possible values: <b>'custom', 'predefinedCheckJobs'</b>.<br> 'predefinedCheckJobs filter' is filter type that is available to all users and covers the most common scenarios for job filtering. There are 4 predefined job filters. Refer to the ‘name’ argument above.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  [\n    {\n      \"id\": \"9d0b43e0-7696-11e3-964b-250efa37a98e\",\n      \"name\": \"my before filter\",\n      \"user\": \"Administrator\",\n      \"query\": \"%7B%22and%22%3A%5B%7B%22initiated%22%3A%7B%22is_before%22%3A%222016-09-30T11%3A18%3A50%22%7D%7D%5D%7D\",\n      \"shared\": {\n        \"groups\": [\"Administrators\"],\n        \"users\": []\n      },\n      \"visibility\": true\n    },\n    {\n      \"id\": \"9d0b43e0-7696-11e3-964b-123456789012\",\n      \"name\": \"my after filter\",\n      \"user\": \"Administrator\",\n      \"query\": \"%7B%22and%22%3A%5B%7B%22initiated%22%3A%7B%22is_after%22%3A%222016-09-30T11%3A18%3A50%22%7D%7D%5D%7D\",\n      \"shared\": {\n        \"groups\": [\"Administrators\"],\n        \"users\": []\n      },\n      \"visibility\": true\n    },\n    {\n        \"id\": \"dcea7ec0-b596-11ec-b314-a13122ab19f3\",\n        \"name\": \"alertJobs\",\n        \"query\": \"%7B%22and%22%3A%5B%7B%22status%22%3A%7B%22is%22%3A%22alert%22%7D%7D%5D%7D\",\n        \"shared\": {\n            \"users\": [],\n            \"groups\": []\n        },\n        \"visibility\": true,\n        \"user\": \"\",\n        \"type\": \"predefinedCheckJobs\"\n    },\n    {\n        \"id\": \"dcee4f50-b596-11ec-b314-a13122ab19f3\",\n        \"name\": \"completedJobs\",\n        \"query\": \"%7B%22and%22%3A%5B%7B%22status%22%3A%7B%22is%22%3A%22completed%22%7D%7D%5D%7D\",\n        \"shared\": {\n            \"users\": [],\n            \"groups\": []\n        },\n        \"visibility\": true,\n        \"user\": \"\",\n        \"type\": \"predefinedCheckJobs\"\n    },\n  ]\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/jobFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/jobFilters?ids=3363eb10-bafc-11e6-83b8-8d274e5d1b3b&fields=id,name,query&includePredefined=true' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobfilters/controller.js",
    "groupTitle": "Job_filters"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/jobFilters/:id?action=share",
    "title": "Share",
    "name": "ShareJobFilter",
    "group": "Job_filters",
    "version": "1.0.0",
    "description": "<p>This method allows to share a user's job filters with other users and groups. Note that it is not allowed to share predefined job filters.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job filter ID. The job filter ID is passed via an URL.</p>"
          },
          {
            "group": "Parameter",
            "type": "Array",
            "optional": true,
            "field": "users",
            "description": "<p>Array of users to share the job filter with</p>"
          },
          {
            "group": "Parameter",
            "type": "Array",
            "optional": true,
            "field": "groups",
            "description": "<p>Array of groups to share the job filter with</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error message. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"users\" : [\"Administrator\", \"Ann\", \"Bert\"],\n  \"groups\": [\"Reviewers\"]\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success",
          "content": "HTTP/1.1 200\n{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden. Returned when the user tries to share a predefined job filter.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found. Returned when the requested job filter is not found.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": \"Unauthorized.\"\n}",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  \"status\": false,\n  \"error\": \"Cannot share the job filter. Error: The requested action is not allowed.\"\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": \"Cannot share the job filter. Error: Cannot find filters.\"\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": \"Cannot edit the job filter.\"\n}",
          "type": "json"
        }
      ]
    },
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobfilters/controller.js",
    "groupTitle": "Job_filters",
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1http://127.0.0.1:51088/api/v1/jobFilters/:id?action=share"
      }
    ]
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/jobFilters/:id?action=unshare",
    "title": "Unshare",
    "name": "UnshareJobFilter",
    "group": "Job_filters",
    "version": "1.0.0",
    "description": "<p>This method stops sharing a user's job filters with other users and groups. Note that it is not allowed to unshare predefined job filters.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Job filter ID. The job filter ID is passed via an URL.</p>"
          },
          {
            "group": "Parameter",
            "type": "Array",
            "optional": true,
            "field": "users",
            "description": "<p>Array of users to stop sharing the job filter with</p>"
          },
          {
            "group": "Parameter",
            "type": "Array",
            "optional": true,
            "field": "groups",
            "description": "<p>Array of groups to stop sharing the job filter with</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error message. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"users\" : [\"Peter\", \"Ann\"],\n  \"groups\": [\"Reviewers\"]\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success",
          "content": "HTTP/1.1 200\n{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden. Returned when the user tries to unshare a predefined job filter.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "404",
            "description": "<p>Not Found. Returned when the requested job filter is not found.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": \"Unauthorized.\"\n}",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden\n{\n  \"status\": false,\n  \"error\": \"Cannot unshare the job filter. Error: The requested action is not allowed.\"\n}",
          "type": "json"
        },
        {
          "title": "404 Not Found:",
          "content": "HTTP/1.1 404 Not Found\n{\n  \"status\": false,\n  \"error\": \"Cannot unshare the job filter. Error: Cannot find filters.\"\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": \"Cannot edit the job filter.\"\n}",
          "type": "json"
        }
      ]
    },
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobfilters/controller.js",
    "groupTitle": "Job_filters",
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1http://127.0.0.1:51088/api/v1/jobFilters/:id?action=unshare"
      }
    ]
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/jobs",
    "title": "List",
    "name": "GetJobs",
    "group": "Jobs",
    "version": "1.0.0",
    "description": "<p>This method returns the list of jobs the user has access to. A number of URL parameters is available to filter, sort, or limit the number of items in the list. Also, the user is able to filter out any unneeded fields.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "fields",
            "description": "<p>The list of fields to return. The list of all the response fields can be found below. If this  parameter is missing, all fields are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "filter",
            "description": "<p>Job filter ID or URL-encoded JSON string representing the Switch filter query. The Switch filter query format is described above in the Add job filter request. This filter will be applied to the returned job list.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "sort",
            "description": "<p>The list of fields to sort on. The default sort order is ascending, a hyphen in front of the field makes the order descending. For example, <i>sort=-size,-updated</i> sorts the result by size from big to small. If the sizes are equal, recently updated jobs are on top. If the sort field is not specified, the default sorting is -updated,status.</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": true,
            "field": "limit",
            "description": "<p>Sets a limit on the number of returned jobs. Default value is 1000. Filtering and sorting are applied before applying the limit.</p>"
          },
          {
            "group": "Parameter",
            "type": "Date",
            "optional": true,
            "field": "lastUpdated",
            "description": "<p>This parameter allows to retrieve jobs that were updated since the specified date and time (in ISO format). Use this parameter to retrieve only the latest job updates instead of all jobs.</p>"
          },
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "data",
            "description": "<p>Set to true if you need a list of jobs according to other parameters in the request. Set to false to get only the information that the job list according to other parameters is not empty. Default is true.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The status of the request. True if the request was processed successfully and the jobs were successfully retrieved (taking into account the data request parameter).</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": false,
            "field": "maxUpdated",
            "description": "<p>This value can be useful to get the latest updates for jobs since the previous request. For example, it is possible to store maxUpdated between two different job list requests and use it as lastUpdated parameter value. In that case, the second call GET / jobs?lastUpdated=<maxUpdated> returns only the newest jobs since the send time of the first call.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": false,
            "field": "data",
            "description": "<p>The value of this field depends on the value of the request parameter &quot;data&quot; (see higher):<br> If set to true (or not provided), an array of jobs is returned. If set to false, two values are possible: true if there's at least one job that matches the request parameters, or false if there are no matching jobs.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.id",
            "description": "<p>ID that can be used to refer to this job in all calls of the Switch Client module API.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.name",
            "description": "<p>Job name.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.flowId",
            "description": "<p>ID of the flow where the job is currently being processed.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.flowName",
            "description": "<p>Name of the flow where the job is currently being processed.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.checkpointId",
            "description": "<p>Checkpoint ID in case the job is currently present in a Checkpoint.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.checkpointName",
            "description": "<p>Checkpoint name in case the job is currently present in a Checkpoint.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.userName",
            "description": "<p>Name of the user the job belongs to.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.family",
            "description": "<p>Processing ID of the parent job where this job originated from. In case the job didn't originate from another job, this will be identical to the processing ID of the job.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.submittedTo",
            "description": "<p>Display name of the Submit point to which the job was submitted.</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": true,
            "field": "data.modificationDate",
            "description": "<p>Modification date of the job (in ISO format).</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": true,
            "field": "data.initiated",
            "description": "<p>Initiated date in ISO format. The date and time when the job reached the first entry point in Switch.</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": false,
            "field": "data.updated",
            "description": "<p>Updated date in ISO format. The date and time when the information about the job was last updated.</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": true,
            "field": "data.onAlertSince",
            "description": "<p>If the job is in alert status, the date and time when the job became alert (reached the Checkpoint it is currently in).</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": true,
            "field": "data.inStateSince",
            "description": "<p>How long job has been in the current state.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.type",
            "description": "<p>Job type (pdf, jpeg etc).</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": true,
            "field": "data.size",
            "description": "<p>Job size.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": true,
            "field": "data.pages",
            "description": "<p>The number of pages for the jobs with type = pdf.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.dimensions",
            "description": "<p>Dimensions for the jobs with type = image.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": true,
            "field": "data.files",
            "description": "<p>Number of files in the job folder or 1 if the job is just a file.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.isFile",
            "description": "<p>true if the job is a file, false if the job is a folder.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.stage",
            "description": "<p>Flow stage.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.state",
            "description": "<p>Job state.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.status",
            "description": "<p>Job status. Possible values:<br> <b>processing</b> - The job is being processed. This status is assigned when the job is submitted to a Submit point or when it leaves a Checkpoint.<br> <b>alert</b> - The job resides in a Checkpoint and needs attention from the client user.<br> <b>completed</b> - The job is processed completely and Switch does not handle it anymore. This status is assigned to the job when it reaches the recycle bin or an end folder and loses its prefix, or when it is sent outside.<br> <b>error</b> - The job failed. Currently this status is only used for submitted jobs that could not be picked up by Switch, for some reason.<br></p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.hasMetadata",
            "description": "<p>If the job is currently in a Checkpoint and the Checkpoint has a metadata definition, hasMetadata is true.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.allowMultipleOutput",
            "description": "<p>If multiple outputs from the Checkpoint are available, allowMultipleOutput is true. Valid only for the jobs that are currently in a Checkpoint and that are available for routing.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.allowReplacing",
            "description": "<p>If the job is currently in a Checkpoint and job replacement is allowed for this job, allowReplacing is true.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.allowReportViewing",
            "description": "<p>If the job is currently in a Checkpoint and report viewing is allowed for this job, allowReportViewing is true.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "data.locked",
            "description": "<p>If the job is currently in a Checkpoint and it is locked for processing, locked is true.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.lockedBy",
            "description": "<p>Name of the user who locked the job.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "data.processingId",
            "description": "<p>Job processing ID that is generated by Switch Server when the job is picked up for processing. This ID can be used to refer to this job in the graphQL reporting API.</p>"
          },
          {
            "group": "Success 200",
            "type": "Array",
            "optional": true,
            "field": "data.outConnections",
            "description": "<p>A Checkpoint's outgoing connections list, in case the job is currently in a Checkpoint.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.outConnections.id",
            "description": "<p>Connection ID.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.outConnections.name",
            "description": "<p>Connection name.</p>"
          },
          {
            "group": "Success 200",
            "type": "Array",
            "optional": true,
            "field": "data.customFields",
            "description": "<p>The custom fields set for the job.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.customFields.name",
            "description": "<p>Custom field name.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.customFields.value",
            "description": "<p>Custom field value.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true,\n  maxUpdated: \"2016-09-30T11:18:50\",\n  data: [\n    {\n      \"id\": \"585134c3bfad3cfc483e3e22\",\n      \"stage\": \"processing point\"\n      \"status\": \"alert\",\n      \"state\": \"preflight\",\n      \"files\": 1,\n      \"isFile\": true,\n      \"modificationDate\": \"2016-09-30T11:18:50\",\n      \"name\": \"Switch.pdf\",\n      \"pages\": 8,\n      \"size\": 3630457,\n      \"type\": \"pdf\",\n      \"initiated\": \"2016-10-12T09:42:29\",\n      \"flowId\": \"1\",\n      \"submittedTo\": \"Submit point 1\",\n      \"userName\": \"Administrator\",\n      \"updated\": \"2016-10-12T09:42:30\",\n      \"family\": \"00001\",\n      \"checkpointId\": \"1\",\n      \"onAlertSince\": \"2016-10-12T09:42:30\",\n      \"checkpointName\": \"Checkpoint 1\",\n      \"hasMetadata\": false,\n      \"allowMultipleOutput\": false,\n      \"allowReplacing\": true,\n      \"customFields\": [\n        {\n          \"name\": \"custom field 1\",\n          \"value\": \"custom value for custom field 1\"\n        }\n      ],\n      \"outConnections\": [\n        {\n          \"id\": \"1\",\n          \"name\": \"Folder 1\"\n        }\n      ],\n      \"inStateSince\": \"2016-10-12T09:42:30\",\n      \"processingId\": \"00001\",\n    },\n  ]\n}",
          "type": "json"
        },
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true,\n  data: true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "400",
            "description": "<p>Bad Request</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; unsupported parameter: unsupported_param'\n}",
          "type": "json"
        },
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; invalid fields param.'\n}",
          "type": "json"
        },
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; unsupported fields param: unsupported_param'\n}",
          "type": "json"
        },
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; invalid sorting field'\n}",
          "type": "json"
        },
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; unsupported sorting field: unsupported_param'\n}",
          "type": "json"
        },
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; invalid parameter value: field_name'\n}",
          "type": "json"
        },
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request\n{\n  status: false,\n  error: 'Bad request; unsupported filtering field: unsupported_field'\n}",
          "type": "json"
        },
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  status: false,\n  error: 'Cannot get the list of the jobs'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/jobs"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/jobs?fields=name,checkpointName,flowName,userName,stage,size&filter=%7B%22and%22%3A%5B%7B%22userName%22%3A%7B%22is%22%3A%22Administrator%22%7D%7D%5D%7D&sort=-size&limit=50&data=true' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/jobs/controller.js",
    "groupTitle": "Jobs"
  },
  {
    "type": "post (graphql)",
    "url": "http://127.0.0.1:51088/api/v1/graphql",
    "title": "GraphQL",
    "version": "1.0.0",
    "name": "dashboardGraphQL",
    "group": "Jobs_info",
    "description": "<p>This method allows you to request various data for the jobs that are currently being processed (processing jobs). It also allows you to request statistics on already processed jobs (<b>only if the Reporting Module is activated</b>). More information is available in the Documentation Explorer below. Click &lt; Docs in the top right corner of the widget to open the Documentation Explorer.<br><br> Interface for creating and testing GraphQL queries. For more information about GraphQL, please check http://graphql.org/learn/</p>",
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/graphql"
      }
    ],
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          },
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-Type",
            "description": "<p>Should be application/graphql</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Headers example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n  \"Content-Type\": \"application/graphql\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "examples": [
        {
          "title": "Query to get all processing jobs in a specific flow and a specific stage",
          "content": "{\n  jobs(filter: \"{\\\"and\\\":[{\\\"flow\\\":{\\\"is\\\":\\\"Stats\\\"}}, {\\\"stage\\\":{\\\"is\\\":\\\"Problem jobs\\\"}}]}\") {\n    count\n    jobs {\n      processingId\n      dimensions\n      files\n      flowId\n      initiated\n      isFile\n      family\n      locked\n      lockedBy\n      name\n      flowName\n      size\n      state\n      stage\n      status\n      type\n      userName\n      flowName\n      checkpointId\n      submittedTo\n      updated\n      inStateSince\n      modificationDate\n      pages\n      onAlertSince\n      rushJobRank\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get all jobs ids in all flows",
          "content": "{\n  jobs(finished: 1) {\n    count\n    jobs {\n      processingId\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get all processing jobs with private data and external metadata",
          "content": "{\n  jobs(filter: \"\") {\n    count\n    jobs {\n      name\n      flow\n      state\n      stage\n      privateData{\n        key\n        value\n      }\n      externalMetadata{\n        datasetName\n        link\n      }\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get processing jobs with filtering on job ID, private data keys and external dataset names",
          "content": "{\n  jobs(filter: \"\", ids:[\"001ET\", \"001ES\", \"001EW\"], keys: [\"count\"], datasets: [\"Submit\"]) {\n    count\n    jobs {\n      name\n      flow\n      state\n      stage\n      privateData{\n        key\n        value\n      }\n      externalMetadata{\n        datasetName\n        link\n      }\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get statistics on processed jobs per flow for the year 2017 grouped by month",
          "content": "{\n  request_alias: jobs_hist(groupBy: [\"flow\"],\n                           since: \"2017-01-01T00:00:00Z\",\n                           to: \"2018-01-01T00:00:00Z\",\n                           period: \"month\" ) {\n    jobs {\n      group {\n        flow\n        period {\n          start\n          finish\n        }\n      }\n      count\n      totalSize\n      avgTimeInFlow\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get active sessions, and sessions that are finished after finishTime",
          "content": "{\n  sessions(finishTime: \"2019-07-09T07:52:43.302Z\") {\n    sessions {\n      userName\n      userFullName\n      userEmail\n      ip\n      startTime\n      finishTime\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get Switch process information",
          "content": "{\n  monitoring {\n    switchProcesses {\n      pid\n      parentPid\n      name\n      cpu\n      memory\n      started\n      command\n    }\n  }\n}",
          "type": "json"
        },
        {
          "title": "Query to get all flows with the flow stages the flows contain and the flow groups the flows belong to",
          "content": "{\n  flows {\n    flows {\n      name\n      stages\n      groups {\n        id\n        name\n      }\n    }\n  }\n}",
          "type": "json"
        }
      ]
    },
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/routes/api.js",
    "groupTitle": "Jobs_info"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:51088/api/v1/messageFilters",
    "title": "Add",
    "name": "AddMessageFilter",
    "group": "Message_filters",
    "version": "1.0.0",
    "description": "<p>This method allows the user to save a message filter in Switch, so it can be retrieved and applied later on.</p> <p>The Switch message filter query is in JSON format, which allows the user to save message filters between sessions.</p> <p>Below you can find some examples of filter queries.</p> <p>Default filter:</p>   <pre> {\"type\":[\"error\",\"warning\",\"info\"]} </pre> <p>Filtering messages by period:</p>   <pre> {\"type\":[\"error\",\"warning\",\"info\"],\"period\":\"1d\"} </pre>   <pre> {\"type\":[\"error\",\"warning\",\"info\"],\"period\":\"1h\"} </pre> <p>Filtering messages by timestamp:</p>   <pre> {\"type\":[\"error\",\"warning\",\"info\"],\"timestamp\":\",2018-01-01T09:00:00.622Z\"} </pre>   <pre> {\"type\":[\"error\",\"warning\",\"info\"],\"timestamp\":\"2018-01-01T09:00:00.622Z\"} </pre>   <pre> {\"type\":[\"error\",\"warning\",\"info\"],\"timestamp\":\"2018-01-01T09:00:00.273Z,2018-01-01T09:30:00.273Z\"} </pre> <p>Filtering messages by flow:</p>   <pre> {\"type\":[\"error\",\"info\",\"warning\"],\"flow\":\"Flow\"} </pre> <p>Filtering messages by type:</p>   <pre> {\"type\":[\"error\",\"no warning\",\"no info\"]} </pre> <p>Filtering messages by module:</p>   <pre> {\"type\":[\"error\",\"info\",\"warning\"],\"module\":\"Folder\"} </pre> <p>Filtering messages by element:</p>   <pre> {\"type\":[\"error\",\"info\",\"warning\"],\"element\":\"Submit point\"} </pre> <p>Filtering messages by prefix:</p>   <pre> {\"type\":[\"error\",\"info\",\"warning\"],\"prefix\":\"0000\"} </pre> <p>Filtering messages by job:</p>   <pre> {\"type\":[\"error\",\"info\",\"warning\"],\"job\":\"file.txt\"} </pre> <p>Filtering messages by message:</p>   <pre> {\"type\":[\"error\",\"info\",\"warning\"],\"message\":\"Moved job to folder\"} </pre>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "name",
            "description": "<p>Filter name.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "query",
            "description": "<p>Filter query in the format described above.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example:",
          "content": "{\n  \"name\" : \"my after filter\",\n  \"query\" : \"{\\\"job\\\": \\\"MyJob\\\", \\\"timestamp\\\": \\\"2017-01-25T10:52:00.234Z,2017-05-25T10:52:00.234Z\\\", \\\"type\\\": \\\"info\\\"}\"\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Automatically generated ID of the just added filter.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  \"id\": \"9d0b43e0-7696-11e3-964b-123456789012\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot add message filter.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/messageFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:51088/api/v1/messageFilters' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01' -H 'Content-Type: application/json' --data '{\"name\": \"filterName\", \"query\"=\"%7B%0A%20%20job%3A%20%22MyJob%22%2C%0A%20%20timestamp%3A%20%222017-01-25T10%3A52%3A00.234Z%2C2017-05-25T10%3A52%3A00.234Z%22%2C%0A%20%20type%3A%20%22info%22%0A%7D\"}'",
        "type": "json"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/messageFilters/controller.js",
    "groupTitle": "Message_filters"
  },
  {
    "type": "delete",
    "url": "http://127.0.0.1:51088/api/v1/messageFilters",
    "title": "Remove",
    "name": "DeleteMessageFilter",
    "group": "Message_filters",
    "version": "1.0.0",
    "description": "<p>This method removes a message filter.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "ids",
            "description": "<p>Comma separated list of filter IDs. If not specified, all filters for the current user will be removed.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success",
          "content": "HTTP/1.1 200\n{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot remove the message filters.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/messageFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X DELETE 'http://127.0.0.1:51088/api/v1/messageFilters?ids=3363eb10-bafc-11e6-83b8-8d274e5d1b3b' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/messageFilters/controller.js",
    "groupTitle": "Message_filters"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:51088/api/v1/messageFilters/:id",
    "title": "Edit",
    "name": "EditMessageFilter",
    "group": "Message_filters",
    "version": "1.0.0",
    "description": "<p>This method allows the user to edit a filter in Switch.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "id",
            "description": "<p>Filter ID. Unlike the other parameters, the filter ID is passed via a URL.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "name",
            "description": "<p>Filter name.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "query",
            "description": "<p>Filter query in the format described above.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used. Unlike all other parameters, lang is passed via a URL.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"name\" : \"my after filter\",\n  \"query\" : \"{\\\"job\\\": \\\"MyJob\\\", \\\"timestamp\\\": \\\"2017-01-25T10:52:00.234Z,2017-05-25T10:52:00.234Z\\\", \\\"type\\\": \\\"info\\\"}\"\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success",
          "content": "HTTP/1.1 200\n{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "String",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": 'Cannot edit message filter.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/messageFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:51088/api/v1/messageFilters/3363eb10-bafc-11e6-83b8-8d274e5d1b3b' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01' -H 'Content-Type: application/json' --data '{\"name\": \"filterName\", \"query\"=\"%7B%0A%20%20job%3A%20%22MyJob%22%2C%0A%20%20timestamp%3A%20%222017-01-25T10%3A52%3A00.234Z%2C2017-05-25T10%3A52%3A00.234Z%22%2C%0A%20%20type%3A%20%22info%22%0A%7D\"}'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/messageFilters/controller.js",
    "groupTitle": "Message_filters"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/messageFilters",
    "title": "List",
    "name": "GetMessageFilter",
    "group": "Message_filters",
    "version": "1.0.0",
    "description": "<p>This call returns the list of message filters for the current user.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "ids",
            "description": "<p>Comma separated list of filter IDs. If not specified, all filters for the current user will be returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "fields",
            "description": "<p>List of fields to return. Supported fields: ID, name, query. If not specified, all fields are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "id",
            "description": "<p>Filter ID automatically assigned by Switch.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "name",
            "description": "<p>Filter name.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "query",
            "description": "<p>Filter query in the format described in Message filters - Add section.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  [\n    {\n      \"id\": \"9d0b43e0-7696-11e3-964b-250efa37a98e\",\n      \"name\": \"my before filter\",\n      \"query\": \"{\\\"job\\\": \\\"MyJob\\\", \\\"timestamp\\\": \\\"2017-01-25T10:52:00.234Z,2017-05-25T10:52:00.234Z\\\", \\\"type\\\": \\\"info\\\"}\"\n    },\n    {\n      \"id\": \"9d0b43e0-7696-11e3-964b-123456789012\",\n      \"name\": \"my after filter\",\n      \"query\": \"{\\\"job\\\": \\\"MyJob\\\", \\\"timestamp\\\": \\\"2017-01-25T10:52:00.234Z,2017-05-25T10:52:00.234Z\\\", \\\"type\\\": \\\"info\\\"}\"\n    }\n  ]\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/messageFilters"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/messageFilters?ids=3363eb10-bafc-11e6-83b8-8d274e5d1b3b&fields=id,name,query' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/messageFilters/controller.js",
    "groupTitle": "Message_filters"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/messages",
    "title": "List",
    "name": "GetMessagesList",
    "group": "Messages",
    "version": "1.0.0",
    "description": "<p>This call returns a list of messages.</p> <p><b>  Remarks: </b></p> <ol> <li> <p>When no 'type' argument is given, only warning, error, and info messages are returned. All other types should be defined separately.</p> </li> <li> <p>For Module, Flow, Element, Message, Prefix and Job, the same type of filter will be used. Multiple values for the same column are always OR filters. E.g:</p> <p><i>GET api/v1/messages?module=foo </i> - The result contains messages with module containing the word foo. <br> <i>GET api/v1/messages?module=foo&amp;module=bar </i> - The result contains messages with module containing the word foo OR bar.</p> </li> </ol> <p><b>  Note: </b> Note: If a <b>filter</b> field is specified all other fields (like <b>period, timestamp, flow, type, module, element, prefix, job, message</b>) will be ignored.</p> <p><b>  Note: </b> All the messages in the response will have a timestamp in UTC, unless there is an HTTP header in the request providing the client timezone. In the latter case the timestamps are converted into the client local time and the timezone is also provided in the timestamp.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "period",
            "description": "<p>Relative date filtering by period. This parameter supports only two possible relative measures: 'days' and 'hours'. <br> <b> period=2d </b> - Retrieves messages logged in the past 2 days <br> <b> period=2h </b> - Retrieves messages logged in the past 2 hours</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "timestamp",
            "description": "<p>Absolute date filtering by timestamp. <br> <b> timestamp=YYYY-MM-DDThh:mmZ </b> - Retrieves messages issued after a certain timestamp. <br> <b> timestamp=,YYYY-MM-DDThh:mmZ </b> - Retrieves messages issued before a certain timestamp. <br> <b> timestamp=YYYY-MM-DDThh:mm:ssZ,YYYY-MM-DDThh:mm:ssZ </b> - Retrieves messages issued between certain timestamps. Note: All the messages in the response will have a timestamp in UTC, unless there is an HTTP header in the request providing the client timezone. In the latter case the timestamps are converted into the client local time and the timezone is also provided in the timestamp.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "flow",
            "description": "<p>Filtering messages by flow.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "type",
            "description": "<p>Filtering messages by type. Possible values: 'error', 'warning', 'info', 'assert', 'debug'. If not specified, warning, error, and info messages are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "module",
            "description": "<p>Filtering messages by module.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "element",
            "description": "<p>Filtering messages by element.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "prefix",
            "description": "<p>Filtering messages by prefix.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "job",
            "description": "<p>Filtering messages by job.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "message",
            "description": "<p>Filtering messages by message.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "filter",
            "description": "<p>Message filter ID or URL-encoded JSON string representing the Switch message filter query. The Switch message filter query format is described in the Add message filter request. This filter will be applied to the returned messages.<br> Note: This filter overrides all other parameters that can be used to filter messages. If specified, all other parameters (like period, timestamp, flow, type, module, element, prefix, job, message) will be ignored.</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": true,
            "field": "limit",
            "description": "<p>Retrieves only N messages; default value is 500. Maximum value is 100000.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "sort",
            "description": "<p>Sorts the returned messages by one of the following fields: <b>  [period, timestamp, flow, type, module, element, prefix, job, message] </b> <br> A hyphen <b> &quot; - &quot; </b> in front of the sort field makes the sort order descending, otherwise it's ascending.<br> To sort on multiple fields, enter a list of comma separated fields, each with a possible unary negative to imply descending sort order e.g. <b>  sort=-time,element... <b/></p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "range",
            "description": "<p>Returns a subset of the retrieved messages based on a set range. The range specifies the sequence number of the message in the response messages list. The range is actually not the same as the message ID; it’s just a rank in the result set according to the current selection. For example, it is possible to request some messages and to extract only 20 of them starting from messages with sequence number 40 and ending with 60. In such case, the request may look like GET /messages?type=error&amp;range=40,60.: <br> <b> range=100 </b> - Retrieves messages starting from sequence number 100 in the selection <br> <b> range=,100 </b> - Retrieves messages starting from sequence number 0 and finishing at sequence number 100 in the selection <br> <b> range=50,100 </b> - Retrieves messages starting from sequence number 50 and finishing at sequence number 100 in the selection <br></p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "id",
            "description": "<p>Returns a subset of the retrieved messages based on their IDs (Note: Message IDs start at zero and increase by one for the next message), for example: <br> <b> id=152 </b> - Retrieves messages since message with ID 152 <br> <b> id=,850 </b> - Retrieves messages up to the message with ID 850 <br> <b> id=15,30 </b> - Retrieves all messages with an ID between 15 and 30 <br></p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return messages and possible errors. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request Params Example :",
          "content": "{\n  <field name>: <field value>,\n  limit: \"500\",\n  sort: \"-type\",\n  range: \"50,100\",\n  lang: \"frFR\"\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "status",
            "description": "<p>Specifies whether or not the messages were retrieved successfully. <br> <b> success</b>: Messages were successfully retrieved. <br> <b> error</b>: Something went wrong during message retrieval.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": true,
            "field": "maxId",
            "description": "<p>Highest messages ID in the messages list at the moment of the request. If there are no messages, the value is equal to -1. The value does not depend on filter or sort settings and is useful to get the latest set of messages since the previous request. For example, it is possible to store the maxId between two different messages list requests and use it as ID parameter value. In such case, the second call GET / messages?id=<maxId> returns only the newest messages since the first call.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": true,
            "field": "messages",
            "description": "<p>Array with messages fields.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "messages.id",
            "description": "<p>Message ID.</p>"
          },
          {
            "group": "Success 200",
            "type": "Date",
            "optional": false,
            "field": "messages.timestamp",
            "description": "<p>Message time.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.type",
            "description": "<p>Message type.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.localizedType",
            "description": "<p>The message type translated into the language specified in the parameter lang.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.module",
            "description": "<p>The module the message is related to.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.prefix",
            "description": "<p>The prefix of the job the message is related to.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.flow",
            "description": "<p>The flow the message is related to.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.element",
            "description": "<p>The flow element the message is related to.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.job",
            "description": "<p>The job the message is related to.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "messages.message",
            "description": "<p>Message.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: \"success\",\n  \"messages\" : [\n     {\n       \"id\": \"1\",\n       \"timestamp\": \"value\",\n       \"type\": \"value\",\n       \"localizedType\": \"value\",\n       \"module\": \"value\",\n       \"prefix\": \"value\",\n       \"flow\": \"value\",\n       \"element\": \"value\",\n       \"job\": \"value\",\n       \"message\": \"value\"\n     },\n   ]\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "Object",
            "optional": false,
            "field": "400",
            "description": "<p>Bad Request</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          },
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "403",
            "description": "<p>Forbidden</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error <br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "400 Bad Request:",
          "content": "HTTP/1.1 400 Bad Request",
          "type": "json"
        },
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "403 Forbidden:",
          "content": "HTTP/1.1 403 Forbidden",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"error\": 'Cannot load messages.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/messages"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/messages' -d \"period=2d+&type=info&module=Switch&message=User&limit=500&sort=-type&range=50%2C100&lang=frFR\" -H 'Authorization: Bearer 146c536dff0abea18ea4d4a3f5d89f721b778b41'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/routes/api.js",
    "groupTitle": "Messages"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:51088/api/v1/messages/clear",
    "title": "Clear log",
    "name": "PostClearMessages",
    "group": "Messages",
    "version": "1.0.0",
    "description": "<p>This method cleans the log messages database.</p>",
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "result",
            "description": "<p>Boolean value response on the operation status (success or not).</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  \"result\": \"true\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error <br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"result\": \"false\",\n  \"error\": 'Cannot clear messages.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/messages/clear"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:51088/api/v1/messages/clear' -H 'Authorization: Bearer b57a90e627d1043be06da8693ab6a8597f3acdb6'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/routes/api.js",
    "groupTitle": "Messages"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/ping",
    "title": "Ping",
    "name": "Ping",
    "group": "Ping",
    "version": "1.0.0",
    "description": "<p>This method pings the Switch Web Services to check their availability, to check the user's status and optionally to refresh the user's session.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "refresh",
            "description": "<p>URL parameter. Flag that shows if the call will only check the user's status or also refresh the user's session.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "data",
            "description": "<p>Status of the current user's activity (0 = active; 1 = about to expire; 2 = invalid)</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"status\": true,\n  \"data\": 1\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/ping"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl 'http://127.0.0.1:51088/api/v1/ping?refresh=true' -H 'Authorization: Bearer c5b6583634cf25e8fbfe7fbe972dc2fc883db953'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/ping/controller.js",
    "groupTitle": "Ping"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/submitpoints/:id",
    "title": "List",
    "name": "GetSubmitpoints",
    "group": "Submit_points",
    "version": "1.0.0",
    "description": "<p>This method returns the Submit points the user has access to.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "id",
            "description": "<p>URL parameter. ID of the Submit point in the format <i> &lt;flowId&gt;-&lt;objectId&gt; </i>. The Submit point ID must contain two parts: the flow ID and the Submit point ID. If no ID is specified, all Submit points are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "fields",
            "description": "<p>Fields which should be included in the response, for example: <i> fields=flowId,flowName,objectId,name,flowState,thumbnail </i>).<br>If no fields are specified, all fields are returned.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "name",
            "description": "<p>Display name of the Submit point.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "descriptionType",
            "description": "<p>Indicates the format in which the description will be displayed. Possible values: &quot;text/plain&quot;, &quot;text/html&quot;.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "description",
            "description": "<p>Display description of the Submit point. The description can be in text or HTML format.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "flowId",
            "description": "<p>Flow ID.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "objectId",
            "description": "<p>Submit Point ID.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "flowName",
            "description": "<p>Flow name.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "flowState",
            "description": "<p>Flow state. Possible values: &quot;Ready&quot;, &quot;Running&quot;, &quot;Starting&quot;, &quot;Scheduled&quot;, &quot;Stopping&quot;, &quot;Unknown&quot;.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "thumbnail",
            "description": "<p>Submit point thumbnail in base64 encoding.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "hasMetadata",
            "description": "<p>True if the user can provide metadata while submitting a job, false otherwise.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "jobRequired",
            "description": "<p>True if the user should submit a job. If set to false, the user can only submit metadata or trigger an action (depending on the configuration of the Submit point).</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": true,
            "field": "metadata",
            "description": "<p>Submit point metadata. The format is similar to the one of job metadata; for more information, refer to the 'Get job metadata' request.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": true,
            "field": "switchHelperRequired",
            "description": "<p>True if the Submit point requires attaching the job origin and/or the creation date to the submitted jobs as done by the Switch Helper.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": true,
            "field": "accept",
            "description": "<p>Allowed job types. Possible values: &quot;filesOnly&quot;, &quot;filesAndFolders&quot;, &quot;foldersOnly&quot;.</p>"
          },
          {
            "group": "Success 200",
            "type": "String[]",
            "optional": true,
            "field": "acceptFileTypes",
            "description": "<p>The list of allowed file extensions.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response: HTTP/1.1 200 OK",
          "content": "{\n  \"flowId\":2,\n  \"objectId\": 3,\n  \"name\":\"Submit point 1\",\n  \"descriptionType\":\"text/plain\",\n  \"description\":\"some description\",\n  \"flowState\": \"Running\",\n  \"flowName\" : \"testFlow\",\n  \"hasMetadata\": true,\n  \"jobRequired\": true,\n  \"accept\": \"filesAndFolders\",\n  \"acceptFileTypes\":\"[\"txt\", \"csv\"]\",\n  \"metadata\": {\n    \"id\": \"spMF_2\",\n    \"displayField\" : true,\n    \"name\" : \"Metadata Field 2\",\n    \"readOnly\" : false,\n    \"rememberLastValue\" : false,\n    \"description\" : \"Enum List\",\n    \"type\" : \"enum:45;576;675;567;\",\n    \"valueIsRequired\" : false,\n    \"format\": \"te(s|x)t\",\n    \"value\": \"45\",\n    \"ignoreDuplicates\": false,\n    \"dependency\": \"spMF_1\",\n    \"dependencyCondition\": \"Equals\",\n    \"dependencyValue\": \"46\"\n  },\n  \"thumbnail\": \"iVBORw0KGgoAAAANSUhEUgAAACAAAAAXCAIAAADlBUW4ibTHqWm08rfz95+MI+j1OXEKtU4EI5rtTlFlLcwsJAND+camxqIiqqYXcLField...\",\n  \"switchHelperRequired\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error <br></p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error { \"status\": false, \"error\": 'Cannot get Submit points.' }",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/submitpoints"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl 'http://127.0.0.1:51088/api/v1/submitpoints/1-2?fields=flowId%2CflowName%2CobjectId%2Cname%2CflowState%2Cthumbnail' -H 'Authorization: Bearer c5b6583634cf25e8fbfe7fbe972dc2fc883db953'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/submitpoints/controller.js",
    "groupTitle": "Submit_points"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:55150/api/v1/defaultApp?:extension",
    "title": "DefaultApp",
    "version": "1.0.0",
    "name": "SwitchHelperDefaultApp",
    "group": "SwitchHelperInfo",
    "description": "<p>Returns the default application that will be used for editing files with the specified extension. Returns an empty string in case the default application could not be found.</p>",
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "extension",
            "description": "<p>The file extension for which the default application will be retrieved.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "defaultApp",
            "description": "<p>The name of the application that will be used for editing files with the specified extension.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true,\n  \"defaultApp\": \"Editor name\"\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/defaultApp?jpeg"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:55150/api/v1/defaultApp?jpeg'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperInfo"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:55150/api/v1/ping",
    "title": "Ping",
    "version": "1.0.0",
    "name": "SwitchHelperPing",
    "group": "SwitchHelperInfo",
    "description": "<p>Checks whether or not Switch Helper is able to respond to requests.</p>",
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "version",
            "description": "<p>The version of the Switch Helper.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true,\n  \"version\": \"4.0\"\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/ping"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:55150/api/v1/ping'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperInfo"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:55150/api/v1/cancelEditJob",
    "title": "Cancel editing",
    "version": "1.0.0",
    "name": "SwitchHelperCancelEditJob",
    "group": "SwitchHelperJob",
    "description": "<p>Resets the inEdit attribute for the specified job.</p>",
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/cancelEditJob?5a1d3337ef1c71dfec2634ca"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:55150/api/v1/cancelEditJob?5a1d3337ef1c71dfec2634ca'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "delete",
    "url": "http://127.0.0.1:55150/api/v1/selectJob",
    "title": "Remove job",
    "version": "1.0.0",
    "name": "SwitchHelperDeleteSelectJob",
    "group": "SwitchHelperJob",
    "description": "<p>Allows the user to remove a certain file or folder from a job selection.</p>",
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Number",
            "optional": false,
            "field": "submitId",
            "description": "<p>The ID of the job selection from which to remove a file or folder.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "path",
            "description": "<p>A file or folder path that must be removed from the job selection.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/selectJob"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X DELETE 'http://127.0.0.1:55150/api/v1/selectJob?submitId=1&path=/path/to/testjob1.txt'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:55150/api/v1/editJob",
    "title": "Edit",
    "version": "1.0.0",
    "name": "SwitchHelperEditJob",
    "group": "SwitchHelperJob",
    "description": "<p>Downloads a specified job, opens it in the default editor and marks it as inEdit.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-Type",
            "description": "<p>application/json.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 500": [
          {
            "group": "Error 500",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false.</p>"
          },
          {
            "group": "Error 500",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Errror:\n{\n  \"status\": false,\n  \"error\": 'HTTP protocol error. 401 Unauthorized.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/editJob"
      }
    ],
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "jobId",
            "description": "<p>Job ID. Can be retrieved using the Switch Web Services List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "jobName",
            "description": "<p>Name that the downloaded job will get.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "swsUrl",
            "description": "<p>Switch Web Services address.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "token",
            "description": "<p>Authorization Bearer token. Can be retrieved using the Switch Web Services login call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Language to be used to return error messages. Possible values: enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:55150/api/v1/editJob' -H 'Content-Type: application/json' --data '{ jobId: \"5a1d3337ef1c71dfec2634ca\", jobName: \"jobname.jpeg\", swsUrl: \"http://127.0.0.1:51088\", token: \"ba9a426ffb4e2d8a904ae6b2e26248e11cc5c6ef\", lang: \"deDE\"}'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:55150/api/v1/jobInEdit",
    "title": "IsInEdit",
    "version": "1.0.0",
    "name": "SwitchHelperJobInEdit",
    "group": "SwitchHelperJob",
    "description": "<p>Checks whether or not the specified job is already in edit mode. Returns true in case the job is in edit mode.</p>",
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/jobInEdit?5a1d3337ef1c71dfec2634ca"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:55150/api/v1/jobInEdit?5a1d3337ef1c71dfec2634ca'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "put",
    "url": "http://127.0.0.1:55150/api/v1/replaceJob",
    "title": "Replace",
    "version": "1.0.0",
    "name": "SwitchHelperReplaceJob",
    "group": "SwitchHelperJob",
    "description": "<p>Replaces a specified job in a Checkpoint.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-Type",
            "description": "<p>application/json.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 500": [
          {
            "group": "Error 500",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false.</p>"
          },
          {
            "group": "Error 500",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Errror:\n{\n  \"status\": false,\n  \"error\": 'Failed to replace job: job was not downloaded for editing.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/replaceJob"
      }
    ],
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "jobId",
            "description": "<p>Job ID. Can be retrieved using the Switch Web Services List jobs call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "swsUrl",
            "description": "<p>Switch Web Services address.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "token",
            "description": "<p>Authorization Bearer token. Can be retrieved using the Switch Web Services login call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Language to be used to return error messages. Possible values: enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN. If not specified, enUS will be used.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "updated",
            "description": "<p>Date in ISO format (e.g. 2017-03-17T14:24:13). This field is used to check the job relevance before replacing.</p>"
          }
        ]
      }
    },
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X PUT 'http://127.0.0.1:55150/api/v1/replaceJob' -H 'Content-Type: application/json' --data '{ jobId: \"5a1d3337ef1c71dfec2634ca\", swsUrl: \"http://127.0.0.1:51088\", token: \"ba9a426ffb4e2d8a904ae6b2e26248e11cc5c6ef\", lang: \"deDE\", \"updated\" : \"2016-11-25T16:58:49\"}'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:55150/api/v1/selectJob",
    "title": "Add job",
    "version": "1.0.0",
    "name": "SwitchHelperSelectJob",
    "group": "SwitchHelperJob",
    "description": "<p>Allows the user to select a job to submit. The result of the call should be used in the Submit call. When you call this method, the system file dialog appears where you can choose a job to submit.</p>",
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Boolean",
            "optional": true,
            "field": "jobFolder",
            "description": "<p>Set to true if you want to select a job folder. Default is false.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "acceptFileTypes",
            "description": "<p>Comma-separated list of allowed file types. If not specified or empty, all file types are allowed.</p>"
          },
          {
            "group": "Parameter",
            "type": "Number",
            "optional": true,
            "field": "submitId",
            "description": "<p>The ID of the job selection to add new files or folders to. If not provided, a new job selection is started for each file or folder that is added.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": false,
            "field": "submits",
            "description": "<p>Set of possible submit requests with their IDs.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "submits.submitId",
            "description": "<p>The ID of the job selection to be used in the Submit call to Switch Helper.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "submits.defaultName",
            "description": "<p>The suggested name of the job to submit, based on the selected file or folder name.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "submits.count",
            "description": "<p>The number of files/folders selected for this submit request.</p>"
          },
          {
            "group": "Success 200",
            "type": "Object[]",
            "optional": false,
            "field": "submits.entries",
            "description": "<p>Set of files and folders selected for this submit request.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "submits.entries.path",
            "description": "<p>File or folder path.</p>"
          },
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "submits.entries.size",
            "description": "<p>File or folder size in bytes.</p>"
          },
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "submits.entries.isFolder",
            "description": "<p>Returns true if the path corresponds to a folder, false otherwise.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true,\n  \"submits\": [\n    {\n      \"defaultName\":\"testjob1.txt\",\n      \"count\":2,\n      \"submitId\":1,\n      \"entries\": [{\n        \"path\":\"/path/to/testjob1.txt\",\n        \"size\":123456\n      },{\n        \"path\":\"/path/to/testjob2.txt\",\n        \"size\":654321\n      }]\n    },\n    {\n      \"defaultName\":\"testjob1.txt\",\n      \"count\":1,\n      \"submitId\":2,\n      \"entries\": [{\n        \"path\":\"/path/to/testjob1.txt\",\n        \"size\":123456\n      }]\n    },\n    {\n      \"defaultName\":\"testjob2.txt\",\n      \"count\":1,\n      \"submitId\":3,\n      \"entries\": [{\n        \"path\":\"/path/to/testjob2.txt\",\n        \"size\":654321\n      }]\n    }\n  ]\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/selectJob"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:55150/api/v1/selectJob'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "post",
    "url": "http://127.0.0.1:55150/api/v1/submitJob",
    "title": "Submit",
    "version": "1.0.0",
    "name": "SwitchHelperSubmitJob",
    "group": "SwitchHelperJob",
    "description": "<p>Submits a previously selected job to a Submit point.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Content-Type",
            "description": "<p>application/json.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "jobId",
            "description": "<p>ID that can be used to refer to this job in all calls of the Switch Client module API.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"status\": true,\n  \"jobId\": \"604f4d59822f58c7c52dff04\"\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 500": [
          {
            "group": "Error 500",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>Always false.</p>"
          },
          {
            "group": "Error 500",
            "type": "String",
            "optional": false,
            "field": "error",
            "description": "<p>Error message.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Errror:\n{\n  \"status\": false,\n  \"error\": 'Failed to submit a job'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/submitJob"
      }
    ],
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Number",
            "optional": false,
            "field": "submitId",
            "description": "<p>The ID of the job selection to submit. It can be retrieved using the selectJob call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "swsUrl",
            "description": "<p>Switch Web Services address.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "token",
            "description": "<p>Authorization Bearer token, which can be retrieved using the Switch Web Services login call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "flowId",
            "description": "<p>ID of the flow where the job should be submitted. It can be retrieved using the Switch Web Services List submit points call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "objectId",
            "description": "<p>ID of the Submit point where the job should be submitted. It can be retrieved using the Switch Web Services List submit points call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "jobName",
            "description": "<p>Job name. This can be a different name than the initial file path (with the appropriate extension if it is not a folder): the job will be renamed on the server according to this parameter. By default the original file/folder name will be used.</p>"
          },
          {
            "group": "Parameter",
            "type": "Object[]",
            "optional": true,
            "field": "metadata",
            "description": "<p>Array of job metadata in Switch metadata format.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.id",
            "description": "<p>Unique ID of the metadata field. It can be retrieved using the List submit points call.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.name",
            "description": "<p>Name of the metadata field.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": false,
            "field": "metadata.value",
            "description": "<p>Value of the metadata field.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Language to be used to return error messages. Possible values: enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN. If not specified, enUS will be used.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Request parameters example :",
          "content": "{\n  \"submitId\": 5,\n  \"swsUrl\": \"http://127.0.0.1:51088\",\n  \"token\": \"ba9a426ffb4e2d8a904ae6b2e26248e11cc5c6ef\",\n  \"flowId\": \"4\",\n  \"objectId\": \"2\",\n  \"jobName\": \"file.txt\",\n  \"metadata\": [{\"id\": \"spMF_1\", \"name\": \"Metadata Field 1\", \"value\": \"value1\"},\n               {\"id\": \"spMF_2\", \"name\": \"Metadata Field 2\", \"value\": \"value2\"}]\n  \"lang\": \"deDE\",\n}",
          "type": "json"
        }
      ]
    },
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X POST 'http://127.0.0.1:55150/api/v1/submitJob' -H 'Content-Type: application/json' --data '{ submitId: 5, swsUrl: \"http://127.0.0.1:51088\", token: \"ba9a426ffb4e2d8a904ae6b2e26248e11cc5c6ef\", flowId: \"4\", objectId: \"2\", jobName: \"file.txt\", lang: \"deDE\"}'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:55150/api/v1/progress",
    "title": "Submit progress",
    "version": "1.0.0",
    "name": "SwitchHelperSubmitProgress",
    "group": "SwitchHelperJob",
    "description": "<p>Returns the progress in percentage of the submit request being executed.</p>",
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:55150/api/v1/progress"
      }
    ],
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "Number",
            "optional": false,
            "field": "submitId",
            "description": "<p>The ID of the job selection currently being submitted. This ID can be retrieved using the selectJob call.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Number",
            "optional": false,
            "field": "value",
            "description": "<p>The current submit progress in percentage.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "200 Success HTTP/1.1 200",
          "content": "{\n  \"value\": 14,\n}",
          "type": "json"
        }
      ]
    },
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:55150/api/v1/progress?submitId=1'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebPortalHelper/Doc/helper.js",
    "groupTitle": "SwitchHelperJob"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/thumbnails",
    "title": "List",
    "name": "GetThumbnails",
    "group": "Thumbnails",
    "version": "1.0.0",
    "description": "<p>This method returns thumbnails for specified jobs. Job IDs are enumerated inside the URL like ?jobIds=585134c3bfad3cfc483e3e22,58513b09bfad3cfc483e3e25.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "jobIds",
            "description": "<p>Comma separated list of job IDs for which to retrieve thumbnails. Jobs IDs can be retrieved using the GraphQL Get jobs call and the List jobs call. Note: Either jobIds or jobProcessingIds should be set for this call but not both.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "jobProcessingIds",
            "description": "<p>Comma separated list of job processing IDs for which to retrieve thumbnails. Job processing IDs can be retrieved using the GraphQL Get jobs call and the List jobs call. Note: Either jobIds or jobProcessingIds should be set for this call but not both.</p>"
          },
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error message. Possible values: <b>  enUS, esES, deDE, frFR, itIT, jaJA, ptBR, zhCN </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request</p>"
          },
          {
            "group": "Success 200",
            "type": "Array",
            "optional": false,
            "field": "data",
            "description": "<p>The list of results</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.id",
            "description": "<p>Job ID in case jobIds is used as request param</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.processingId",
            "description": "<p>Processing job ID in case jobProcessingIds is used as request param</p>"
          },
          {
            "group": "Success 200",
            "type": "String",
            "optional": false,
            "field": "data.thumbnail",
            "description": "<p>Base64 encoded thumbnail</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  \"status\": true,\n  \"data\": [\n    {\n      \"id\": \"585134c3bfad3cfc483e3e22\",\n      \"thumbnail\": '/9j/4AAQSkZJRgABBgAAD...WwtFgtFFFZFH/9k=',\n    },\n    {\n      \"id\": \"58513b09bfad3cfc483e3e25\",\n      \"thumbnail\": '/9j/AgGdbjSkZJRjAYABg...jdtAgGdjlZFH/9k=',\n    }\n  ]\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ],
        "Error 5xx": [
          {
            "group": "Error 5xx",
            "type": "Object",
            "optional": false,
            "field": "500",
            "description": "<p>Internal Server Error</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized",
          "type": "json"
        },
        {
          "title": "500 Internal Server Error:",
          "content": "HTTP/1.1 500 Internal Server Error\n{\n  \"status\": false,\n  \"error\": \"Internal server error.\"\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/thumbnails"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/thumbnails?jobIds=585134c3bfad3cfc483e3e22,58513b09bfad3cfc483e3e25' -H 'Authorization: Bearer c5b6583634cf25e8fbfe7fbe972dc2fc883db953'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/thumbnails/controller.js",
    "groupTitle": "Thumbnails"
  },
  {
    "type": "get",
    "url": "http://127.0.0.1:51088/api/v1/userpermissions/groups",
    "title": "List groups",
    "name": "GetGroups",
    "group": "Users",
    "version": "1.0.0",
    "description": "<p>This call returns the list of groups the current user belongs to. In case of an administrator user, all groups are returned.</p>",
    "header": {
      "fields": {
        "Header": [
          {
            "group": "Header",
            "type": "String",
            "optional": false,
            "field": "Authorization",
            "description": "<p>Bearer token</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Header-Example:",
          "content": "{\n  \"Authorization\": \"Bearer b767c71faba9ff7b54e530a0cbf996ddb315131d\"\n}",
          "type": "json"
        }
      ]
    },
    "parameter": {
      "fields": {
        "Parameter": [
          {
            "group": "Parameter",
            "type": "String",
            "optional": true,
            "field": "lang",
            "description": "<p>Indicates which language will be used to return error messages. Possible values: <b>  enUS, frFR, deDE, itIT, zhCN, jaJA </b>. If not specified, enUS will be used.</p>"
          }
        ]
      }
    },
    "success": {
      "fields": {
        "Success 200": [
          {
            "group": "Success 200",
            "type": "Boolean",
            "optional": false,
            "field": "status",
            "description": "<p>The result of the request.</p>"
          },
          {
            "group": "Success 200",
            "type": "Array",
            "optional": false,
            "field": "groups",
            "description": "<p>The list of groups the current user belongs to.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "Success-Response:",
          "content": "HTTP/1.1 200 OK\n{\n  status: true,\n  groups: [\"My group 1\", \"My group 2\"]\n}",
          "type": "json"
        }
      ]
    },
    "error": {
      "fields": {
        "Error 4xx": [
          {
            "group": "Error 4xx",
            "type": "String",
            "optional": false,
            "field": "401",
            "description": "<p>Unauthorized. Note that no JSON is returned in this case.</p>"
          }
        ]
      },
      "examples": [
        {
          "title": "401 Unauthorized:",
          "content": "HTTP/1.1 401 Unauthorized\n{\n  \"status\": false,\n  \"error\": 'Unauthorized.'\n}",
          "type": "json"
        }
      ]
    },
    "sampleRequest": [
      {
        "url": "http://127.0.0.1:51088/api/v1/userpermissions/groups"
      }
    ],
    "examples": [
      {
        "title": "Example usage:",
        "content": "curl -X GET 'http://127.0.0.1:51088/api/v1/userpermissions/groups' -H 'Authorization: Bearer 71f5167b1135f04d423bb63ddcafaba4fb770b01'",
        "type": "curl"
      }
    ],
    "filename": "/Volumes/Workspaces/Switch_24.1_arm64_Build/Depot/SwitchWebService/Sources/src/api/userPermissions/controller.js",
    "groupTitle": "Users"
  }
] });
